package cn.morethank.open.admin.system.service.impl;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.constant.RedisKeyConstant;
import cn.morethank.open.admin.common.exception.ServiceException;
import cn.morethank.open.admin.common.service.RedisService;
import cn.morethank.open.admin.common.util.Convert;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysConfig;
import cn.morethank.open.admin.system.mapper.SysConfigMapper;
import cn.morethank.open.admin.system.service.SysConfigService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;

/**
 * 参数配置表 服务实现类
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@AllArgsConstructor
@Service
public class SysConfigServiceImpl extends ServiceImpl<SysConfigMapper, SysConfig> implements SysConfigService {

    private final SysConfigMapper sysConfigMapper;
    private final RedisService redisService;

    @Override
    public IPage<SysConfig> selectListPage(Page<SysConfig> page, LambdaQueryWrapper<SysConfig> query) {
        return sysConfigMapper.selectPage(page, query);
    }

    @Override
    public int deleteConfigByIds(Long[] configIds) {
        for (Long configId : configIds) {
            SysConfig config = getById(configId);
            if (StringUtils.equals(GlobalConstant.YES, config.getConfigType())) {
                throw new ServiceException(String.format("内置参数【%1$s】不能删除 ", config.getConfigKey()));
            }
            sysConfigMapper.deleteById(configId);
            redisService.del(getCacheKey(config.getConfigKey()));
        }
        return configIds.length;
    }

    @Override
    public void resetConfigCache() {
        clearConfigCache();
        loadingConfigCache();
    }

    /**
     * 清空参数缓存数据
     */
    @Override
    public void clearConfigCache() {
        Collection<String> keys = redisService.keys(RedisKeyConstant.SYS_CONFIG_KEY + "*");
        redisService.deleteObjects(keys);
    }

    /**
     * 加载参数缓存数据
     */
    @Override
    public void loadingConfigCache() {
        QueryWrapper<SysConfig> queryWrapper = new QueryWrapper<>();
        List<SysConfig> configsList = sysConfigMapper.selectList(queryWrapper);
        for (SysConfig config : configsList) {
            redisService.set(getCacheKey(config.getConfigKey()), config.getConfigValue());
        }
    }

    @Override
    public boolean checkConfigKeyUnique(SysConfig config) {
        LambdaQueryWrapper<SysConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysConfig::getConfigKey, config.getConfigKey());
        if(config.getConfigId() != null && config.getConfigId() > 0L) {
            wrapper.ne(SysConfig::getConfigId, config.getConfigId());
        }
        long count = sysConfigMapper.selectCount(wrapper);
        return count > 0;
    }

    @Override
    public Object selectConfigByKey(String configKey) {
        String configValue = Convert.toStr(redisService.get(getCacheKey(configKey)));
        if (StringUtils.isNotEmpty(configValue)) {
            return configValue;
        }
        SysConfig config = new SysConfig();
        config.setConfigKey(configKey);

        LambdaQueryWrapper<SysConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(SysConfig::getConfigKey, config.getConfigKey());
        SysConfig retConfig = sysConfigMapper.selectOne(wrapper);
        if (StringUtils.isNotNull(retConfig)) {
            redisService.set(getCacheKey(configKey), retConfig.getConfigValue());
            return retConfig.getConfigValue();
        }
        return GlobalConstant.EMPTY;
    }

    /**
     * 设置cache key
     *
     * @param configKey 参数键
     * @return 缓存键key
     */
    private String getCacheKey(String configKey) {
        return RedisKeyConstant.SYS_CONFIG_KEY + configKey;
    }
}
