package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.Convert;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysGenColumn;
import cn.morethank.open.admin.system.domain.SysGenTable;
import cn.morethank.open.admin.system.service.SysGenColumnService;
import cn.morethank.open.admin.system.service.SysGenTableService;
import cn.morethank.open.admin.common.util.GenConstants;
import com.alibaba.excel.EasyExcel;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 代码生成 前端控制器
 *
 * @author morethank
 * @since 2023-01-08 00:30:18
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/gen")
public class ToolGenTableController {

    private final SysGenTableService sysGenTableService;
    private final SysGenColumnService sysGenColumnService;
    private final JwtService jwtService;

    /**
     * 详情页
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:gen:query')")
    @GetMapping("{id}")
    public Result detail(@PathVariable Long id) {
        SysGenTable sysGenTable = sysGenTableService.getById(id);
        setTableFromOptions(sysGenTable);

        List<SysGenTable> tables = sysGenTableService.selectGenTableAll();
        List<SysGenColumn> list = sysGenColumnService.selectGenColumnListByTableId(id);
        if(CollectionUtils.isEmpty(list)) {
            // 根据表自动生成数据
            sysGenColumnService.genColumnListByTableName(id, sysGenTable.getTableName());
            // 重新查一遍
            list = sysGenColumnService.selectGenColumnListByTableId(id);
        }
        Map<String, Object> map = new HashMap<>();
        map.put("info", sysGenTable);
        map.put("rows", list);
        map.put("tables", tables);
        return Result.success(map);
    }

    /**
     * 设置代码生成其他选项值
     *
     * @param genTable 设置后的生成对象
     */
    private void setTableFromOptions(SysGenTable genTable) {
        JSONObject paramsObj = JSON.parseObject(genTable.getOptions());
        if (StringUtils.isNotNull(paramsObj)) {
            String treeCode = paramsObj.getString(GenConstants.TREE_CODE);
            String treeParentCode = paramsObj.getString(GenConstants.TREE_PARENT_CODE);
            String treeName = paramsObj.getString(GenConstants.TREE_NAME);
            String parentMenuId = paramsObj.getString(GenConstants.PARENT_MENU_ID);
            String parentMenuName = paramsObj.getString(GenConstants.PARENT_MENU_NAME);

            genTable.setTreeCode(treeCode);
            genTable.setTreeParentCode(treeParentCode);
            genTable.setTreeName(treeName);
            genTable.setParentMenuId(parentMenuId);
            genTable.setParentMenuName(parentMenuName);
        }
    }

    /**
     * 分页查询
     *
     * @param sysGenTable    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:gen:list')")
    @GetMapping(value = "/list")
    public Result list(SysGenTable sysGenTable,
                       @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                       @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询toolGenTable,检索参数toolGenTable={},分页参数pageNo={},pageSize={}", sysGenTable, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysGenTable> query = getQueryWrapper(sysGenTable);
            IPage<SysGenTable> pageList = sysGenTableService.selectListPage(new Page<>(pageNo, pageSize), query);
            // 如果第一页没有数据, 则自动生成数据并重新查询
            if(pageNo == 1 && pageList.getTotal() == 0) {
                // 根据表自动生成数据
                sysGenTableService.genTableList();
                pageList = sysGenTableService.selectListPage(new Page<>(pageNo, pageSize), query);
            }
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 新增保存
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.INSERT)
    @PreAuthorize("@auth.hasAuthority('system:gen:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysGenTable sysGenTable) {
        /*if (toolGenTableService.checkNameUnique(toolGenTable)) {
            return Result.fail("新增代码生成业务'" + toolGenTable.getName() + "'失败，代码生成业务已存在");
        }*/
        sysGenTable.setCreateBy(jwtService.getUserName());
        sysGenTable.setCreateTime(LocalDateTime.now());
        return Result.success(sysGenTableService.save(sysGenTable));
    }

    /**
     * 修改保存
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:gen:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysGenTable sysGenTable) {
        /*if (toolGenTableService.checkNameUnique(toolGenTable)) {
            return Result.fail("修改代码生成业务'" + toolGenTable.getName() + "'失败，代码生成业务已存在");
        }*/
        sysGenTable.setUpdateBy(jwtService.getUserName());
        sysGenTable.setUpdateTime(LocalDateTime.now());
        sysGenTableService.updateGenTable(sysGenTable);
//        toolGenTableService.updateById(toolGenTable);

        return Result.success();
    }

    /**
     * 删除接口
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:gen:remove')")
    @DeleteMapping("/{ids}")
    public Result remove(@PathVariable Long[] ids) {
        return Result.success(sysGenTableService.deleteByIds(ids));
    }

    /**
     * 导出接口
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:gen:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysGenTable sysGenTable) throws Exception {
        LambdaQueryWrapper<SysGenTable> query = getQueryWrapper(sysGenTable);
        List<SysGenTable> list = sysGenTableService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("代码生成业务", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysGenTable.class).sheet("代码生成业务").doWrite(list);
    }

    /**
     * 预览代码
     */
    @OperateLog(title = "代码生成业务", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:gen:preview')")
    @GetMapping("/preview/{tableId}")
    public Result preview(@PathVariable("tableId") Long tableId) throws IOException {
        Map<String, String> dataMap = sysGenTableService.previewCode(tableId);
        return Result.success(dataMap);
    }

    /**
     * 同步数据库
     */
    @PreAuthorize("@auth.hasAuthority('system:gen:edit')")
    @OperateLog(title = "同步数据库", businessType = BusinessType.UPDATE)
    @GetMapping("/synchDb/{tableName}")
    public Result synchDb(@PathVariable("tableName") String tableName) {
        sysGenTableService.synchDb(tableName);
        return Result.success();
    }

    /**
     * 批量生成代码
     */
    @PreAuthorize("@auth.hasAuthority('system:gen:code')")
    @OperateLog(title = "代码生成", businessType = BusinessType.GENCODE)
    @GetMapping("/batchGenCode")
    public void batchGenCode(HttpServletResponse response, String tables) throws IOException {
        String[] tableNames = Convert.toStrArray(tables);
        byte[] data = sysGenTableService.downloadCode(tableNames);
        genCode(response, data);
    }

    /**
     * 生成zip文件
     */
    private void genCode(HttpServletResponse response, byte[] data) throws IOException {
        response.reset();
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.addHeader("Access-Control-Expose-Headers", "Content-Disposition");
        response.setHeader("Content-Disposition", "attachment; filename=\"openadmin.zip\"");
        response.addHeader("Content-Length", "" + data.length);
        response.setContentType("application/octet-stream; charset=UTF-8");
        IOUtils.write(data, response.getOutputStream());
    }

    private LambdaQueryWrapper<SysGenTable> getQueryWrapper(SysGenTable sysGenTable) {
        LambdaQueryWrapper<SysGenTable> query = new LambdaQueryWrapper<>();
        // TODO 添加条件
        if (StrUtil.isNotEmpty(sysGenTable.getTableName())) {
            query.like(SysGenTable::getTableName, sysGenTable.getTableName());
        }
        if (StrUtil.isNotEmpty(sysGenTable.getRemark())) {
            query.like(SysGenTable::getRemark, sysGenTable.getRemark());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, sysGenTable.getParams());
        return query;
    }
}
