package cn.morethank.open.admin.system.controller;

import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.config.AdminConfig;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.LoginAccount;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.domain.UserObject;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.FileUploadUtils;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysDept;
import cn.morethank.open.admin.system.domain.SysRole;
import cn.morethank.open.admin.system.domain.SysUser;
import cn.morethank.open.admin.system.service.SysDeptService;
import cn.morethank.open.admin.system.service.SysPostService;
import cn.morethank.open.admin.system.service.SysRoleService;
import cn.morethank.open.admin.system.service.SysUserService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.plugins.pagination.PageDTO;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 用户管理 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/user")
public class SysUserController {

    private final SysUserService sysUserService;
    private final SysDeptService sysDeptService;
    private final SysRoleService sysRoleService;
    private final SysPostService sysPostService;
    private final JwtService jwtService;

    /**
    * 详情页, 编辑页面需要调用详情接口
    */
    @OperateLog(title = "用户管理", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:user:query')")
    @GetMapping(value = { "/", "/{id}" })
    public Result detail(@PathVariable(value = "id", required = false) Long id) {
        // 查询所有的角色, 按role_sort字段排序
        LambdaQueryWrapper<SysRole> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.orderByAsc(SysRole::getRoleSort);
        List<SysRole> roles = sysRoleService.list(queryWrapper);

        Map<String, Object> dataMap = new HashMap<>();
        dataMap.put("roles", SysUser.isAdmin(id) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        dataMap.put("posts", sysPostService.list());
        if (StringUtils.isNotNull(id)) {
            SysUser sysUser = sysUserService.selectUserById(id);
            dataMap.put("user", sysUser);
            dataMap.put("postIds", sysPostService.selectPostListByUserId(id));
            dataMap.put("roleIds", sysUser.getRoles().stream().map(SysRole::getRoleId).collect(Collectors.toList()));
        }
        return Result.success(dataMap);
    }

    /**
     * 分页查询
     *
     * @param sysUser    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:user:list')")
    @GetMapping(value = "/list")
    public Result list(SysUser sysUser,
                       @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                       @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysUser,检索参数sysUser={},分页参数pageNo={},pageSize={}", sysUser, pageNo, pageSize);
        try {
            // 构造分页查询条件
            Page<SysUser> page = PageDTO.of(pageNo, pageSize);
            IPage<SysUser> pageList = sysUserService.selectListPage(page, sysUser);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 删除用户
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:user:remove')")
    @DeleteMapping("/{userIds}")
    public Result remove(@PathVariable Long[] userIds) {
        Long loginUserId = jwtService.getLoginAccount().getUserId();
        if (ArrayUtils.contains(userIds, loginUserId)) {
            return Result.fail("当前用户不能删除");
        }
        return Result.success(sysUserService.deleteUserByIds(userIds));
    }

    /**
     * 新增用户
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.INSERT)
    @PreAuthorize("@auth.hasAuthority('system:user:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysUser user) {
        if (sysUserService.checkUserNameUnique(user.getUserName())) {
            return Result.fail("新增用户'" + user.getUserName() + "'失败，登录账号已存在");
        } else if (StringUtils.isNotEmpty(user.getPhoneNumber()) && sysUserService.checkPhoneUnique(user)) {
            return Result.fail("新增用户'" + user.getUserName() + "'失败，手机号码已存在");
        } else if (StringUtils.isNotEmpty(user.getEmail()) && sysUserService.checkEmailUnique(user)) {
            return Result.fail("新增用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        user.setCreateBy(jwtService.getUserName());
        user.setCreateTime(LocalDateTime.now());
        return Result.success(sysUserService.insertUser(user));
    }

    /**
     * 修改用户
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:user:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysUser user) {
        sysUserService.checkUserAllowed(user);
        if (StringUtils.isNotEmpty(user.getPhoneNumber()) && sysUserService.checkPhoneUnique(user)) {
            return Result.fail("修改用户'" + user.getUserName() + "'失败，手机号码已存在");
        } else if (StringUtils.isNotEmpty(user.getEmail()) && sysUserService.checkEmailUnique(user)) {
            return Result.fail("修改用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        user.setUpdateBy(jwtService.getUserName());
        user.setUpdateTime(LocalDateTime.now());
        return Result.success(sysUserService.updateUser(user));
    }

    /**
     * 重置密码
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:user:resetPwd')")
    @PutMapping("/resetPwd")
    public Result resetPwd(@RequestBody SysUser user) {
        sysUserService.checkUserAllowed(user);

        user.setUpdateBy(jwtService.getUserName());
        user.setUpdateTime(LocalDateTime.now());
        return Result.success(sysUserService.resetPwd(user));
    }

    /**
     * 获取部门树列表
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:user:list')")
    @GetMapping("/deptTree")
    public Result deptTree(SysDept dept) {
        return Result.success(sysDeptService.selectDeptTreeList(dept));
    }
    
    /**
     * 根据用户编号获取授权角色
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:user:query')")
    @GetMapping("/authRole/{userId}")
    public Result authRole(@PathVariable("userId") Long userId) {
        SysUser user = sysUserService.selectUserById(userId);
        List<SysRole> roles = sysRoleService.selectRolesByUserId(userId);
        Map<String, Object> data = new HashMap<>(2);
        data.put("user", user);
        data.put("roles", SysUser.isAdmin(userId) ? roles : roles.stream().filter(r -> !r.isAdmin()).collect(Collectors.toList()));
        return Result.success(data);
    }

    /**
     * 用户授权角色
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.GRANT)
    @PreAuthorize("@auth.hasAuthority('system:user:edit')")
    @PutMapping("/authRole")
    public Result insertAuthRole(Long userId, Long[] roleIds) {
        sysUserService.insertUserAuth(userId, roleIds);
        return Result.success();
    }

    /**
     * 状态修改
     */
    @PreAuthorize("@auth.hasAuthority('system:user:edit')")
    @OperateLog(title = "用户管理", businessType = BusinessType.UPDATE)
    @PutMapping("/changeStatus")
    public Result changeStatus(@RequestBody SysUser user) {
        sysUserService.checkUserAllowed(user);
        user.setUpdateBy(jwtService.getUserName());
        user.setUpdateTime(LocalDateTime.now());
        return Result.success(sysUserService.updateUser(user));
    }

    @OperateLog(title = "用户管理", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:user:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysUser user) throws IOException {
        List<SysUser> list = sysUserService.selectUserList(user);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("用户", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysUser.class).sheet("用户").doWrite(list);
    }

    /**
     * 个人信息
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:user:query')")
    @GetMapping("/profile")
    public Result profile(HttpServletRequest request) {
        LoginAccount loginAccount = jwtService.getLoginAccount(request);
        UserObject user = loginAccount.getUser();
        Map<String, Object> ajax = new HashMap<>(3);
        ajax.put("user", user);
        ajax.put("roleGroup", sysUserService.selectUserRoleGroup(user.getUserName()));
        ajax.put("postGroup", sysUserService.selectUserPostGroup(user.getUserName()));
        return Result.success(ajax);
    }

    /**
     * 修改用户
     */
    @OperateLog(title = "个人信息", businessType = BusinessType.UPDATE)
    @PutMapping("/profile")
    public Result updateProfile(@RequestBody SysUser user, HttpServletRequest request) {
        LoginAccount loginUser = jwtService.getLoginAccount(request);
        UserObject userObject = loginUser.getUser();
        user.setUserName(userObject.getUserName());
        if (StringUtils.isNotEmpty(user.getPhoneNumber()) && sysUserService.checkPhoneUnique(user)) {
            return Result.fail("修改用户'" + user.getUserName() + "'失败，手机号码已存在");
        } else if (StringUtils.isNotEmpty(user.getEmail()) && sysUserService.checkEmailUnique(user)) {
            return Result.fail("修改用户'" + user.getUserName() + "'失败，邮箱账号已存在");
        }
        user.setUserId(userObject.getUserId());
        user.setPassword(null);
        user.setAvatar(null);
        user.setDeptId(null);

        user.setUpdateBy(jwtService.getUserName());
        user.setUpdateTime(LocalDateTime.now());
        if (sysUserService.updateUser(user) > 0) {
            // 更新缓存用户信息
            userObject.setNickName(user.getNickName());
            userObject.setPhoneNumber(user.getPhoneNumber());
            userObject.setEmail(user.getEmail());
            userObject.setSex(user.getSex());
            jwtService.setLoginAccount(loginUser);
            return Result.success();
        }
        return Result.fail("修改个人信息异常，请联系管理员");
    }

    /**
     * 头像上传
     */
    @OperateLog(title = "用户头像", businessType = BusinessType.UPDATE)
    @PostMapping("/profile/avatar")
    public Result avatar(@RequestParam("avatarfile") MultipartFile file, HttpServletRequest request) throws Exception {
        if (!file.isEmpty()) {
            LoginAccount loginUser = jwtService.getLoginAccount(request);
            String avatar = FileUploadUtils.upload(AdminConfig.getAvatarPath(), file, GlobalConstant.IMAGE_EXTENSION);
            if (sysUserService.updateUserAvatar(loginUser.getUsername(), avatar)) {
                // 更新缓存用户头像
                loginUser.getUser().setAvatar(avatar);
                jwtService.setLoginAccount(loginUser);
                return Result.success(avatar);
            }
        }
        return Result.fail("上传图片异常，请联系管理员");
    }
}
