package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.domain.LoginAccount;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysDept;
import cn.morethank.open.admin.system.domain.SysRole;
import cn.morethank.open.admin.system.domain.SysUser;
import cn.morethank.open.admin.system.domain.SysUserRole;
import cn.morethank.open.admin.system.service.SysDeptService;
import cn.morethank.open.admin.common.service.SysPermissionService;
import cn.morethank.open.admin.system.service.SysRoleService;
import cn.morethank.open.admin.system.service.SysUserService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.plugins.pagination.PageDTO;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 角色管理 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/role")
public class SysRoleController {

    private final SysUserService sysUserService;
    private final SysRoleService sysRoleService;
    private final SysDeptService sysDeptService;
    private final SysPermissionService sysPermissionService;
    private final JwtService jwtService;

    /**
    * 详情页
    */
    @OperateLog(title = "角色管理", businessType = BusinessType.DETAIL)
    @RequestMapping("{id}")
    @PreAuthorize("@auth.hasAuthority('system:role:query')")
    public Result detail(@PathVariable Long id) {
        SysRole sysRole = sysRoleService.getById(id);
        return Result.success(sysRole);
    }

    /**
     * 分页查询
     *
     * @param sysRole  分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:role:query')")
    @GetMapping(value = "/list")
    public Result list(SysRole sysRole, @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysRole,检索参数sysRole={},分页参数pageNo={},pageSize={}", sysRole, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysRole> query = getQueryWrapper(sysRole);
            IPage<SysRole> pageList = sysRoleService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 修改状态
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:role:edit')")
    @PutMapping("/changeStatus")
    public Result changeStatus(@RequestBody SysRole role) {
        sysRoleService.checkRoleAllowed(role);
        return Result.success(sysRoleService.updateRoleStatus(role));
    }

    /**
     * 删除角色
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:role:remove')")
    @DeleteMapping("/{roleIds}")
    public Result remove(@PathVariable Long[] roleIds) {
        return Result.success(sysRoleService.deleteRoleByIds(roleIds));
    }

    /**
     * 新增保存角色
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.INSERT)
    @PreAuthorize("@auth.hasAuthority('system:role:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysRole role) {
        if (sysRoleService.checkRoleNameUnique(role)) {
            return Result.fail("新增角色'" + role.getRoleName() + "'失败，角色名称已存在");
        } else if (sysRoleService.checkRoleKeyUnique(role)) {
            return Result.fail("新增角色'" + role.getRoleName() + "'失败，角色权限已存在");
        }
        role.setCreateBy(jwtService.getUserName());
        role.setCreateTime(LocalDateTime.now());
        return Result.success(sysRoleService.insertRole(role));
    }

    /**
     * 修改保存角色
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:role:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysRole role) {
        sysRoleService.checkRoleAllowed(role);
        if (sysRoleService.checkRoleNameUnique(role)) {
            return Result.fail("修改角色'" + role.getRoleName() + "'失败，角色名称已存在");
        } else if (sysRoleService.checkRoleKeyUnique(role)) {
            return Result.fail("修改角色'" + role.getRoleName() + "'失败，角色权限已存在");
        }
        role.setUpdateBy(jwtService.getUserName());
        role.setUpdateTime(LocalDateTime.now());

        if (sysRoleService.updateRole(role) > 0) {
            // 更新缓存用户权限
            LoginAccount loginUser = jwtService.getLoginAccount();
            if (StringUtils.isNotNull(loginUser) && !loginUser.getUser().isAdmin()) {
                loginUser.setPermissions(sysPermissionService.getMenuPermission(loginUser.getUser()));
                loginUser.setUser(sysUserService.getByUsername(loginUser.getUsername()));
                jwtService.setLoginAccount(loginUser);
            }
            return Result.success();
        }
        return Result.fail("修改角色'" + role.getRoleName() + "'失败，请联系管理员");
    }
    
    /**
     * 查询已分配用户角色列表
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:role:list')")
    @GetMapping("/authUser/allocatedList")
    public Result allocatedList(SysUser user,
                                @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        Page<SysUser> page = PageDTO.of(pageNo, pageSize);
        IPage<SysUser> list = sysUserService.selectAllocatedList(page, user);
        return Result.success(list);
    }

    /**
     * 查询未分配用户角色列表
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:role:list')")
    @GetMapping("/authUser/unallocatedList")
    public Result unallocatedList(SysUser user,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        Page<SysUser> page = PageDTO.of(pageNo, pageSize);
        IPage<SysUser> list = sysUserService.selectUnallocatedList(page, user);
        return Result.success(list);
    }

    /**
     * 批量选择用户授权
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.GRANT)
    @PreAuthorize("@auth.hasAuthority('system:role:edit')")
    @PutMapping("/authUser/selectAll")
    public Result selectAuthUserAll(Long roleId, Long[] userIds) {
        return Result.success(sysRoleService.insertAuthUsers(roleId, userIds));
    }

    /**
     * 取消授权用户
     */
    @OperateLog(title = "角色管理", businessType = BusinessType.GRANT)
    @PreAuthorize("@auth.hasAuthority('system:role:edit')")
    @PutMapping("/authUser/cancel")
    public Result cancelAuthUser(@RequestBody SysUserRole userRole) {
        return Result.success(sysRoleService.deleteAuthUser(userRole));
    }

    @OperateLog(title = "角色管理", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:role:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysRole role) throws Exception {
        List<SysRole> list = sysRoleService.selectRoleList(role);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("角色", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysRole.class).sheet("角色").doWrite(list);
    }

    /**
     * 获取对应角色部门树列表
     */
    @PreAuthorize("@auth.hasAuthority('system:role:query')")
    @GetMapping(value = "/deptTree/{roleId}")
    public Result deptTree(@PathVariable("roleId") Long roleId) {
        Map<String, Object> result = new HashMap<>();
        result.put("checkedKeys", sysDeptService.selectDeptListByRoleId(roleId));
        result.put("depts", sysDeptService.selectDeptTreeList(new SysDept()));
        return Result.success(result);
    }

    /**
     * 批量取消授权用户
     */
    @PreAuthorize("@auth.hasAuthority('system:role:edit')")
    @OperateLog(title = "角色管理", businessType = BusinessType.GRANT)
    @PutMapping("/authUser/cancelAll")
    public Result cancelAuthUserAll(Long roleId, Long[] userIds) {
        return Result.success(sysRoleService.deleteAuthUsers(roleId, userIds));
    }

    private LambdaQueryWrapper<SysRole> getQueryWrapper(SysRole sysRole) {
        LambdaQueryWrapper<SysRole> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysRole.getRoleName())) {
            query.like(SysRole::getRoleName, sysRole.getRoleName());
        }
        if (StrUtil.isNotEmpty(sysRole.getRoleKey())) {
            query.like(SysRole::getRoleKey, sysRole.getRoleKey());
        }
        if (StrUtil.isNotEmpty(sysRole.getStatus())) {
            query.eq(SysRole::getStatus, sysRole.getStatus());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, sysRole.getParams());
        // 排序
        query.orderByAsc(SysRole::getRoleSort);
        return query;
    }
}
