package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.system.domain.SysPost;
import cn.morethank.open.admin.system.service.SysPostService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 岗位管理 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/post")
public class SysPostController {

    private final SysPostService sysPostService;
    private final JwtService jwtService;

    /**
    * 详情页
    */
    @OperateLog(title = "岗位管理", businessType = BusinessType.DETAIL)
    @GetMapping("{id}")
    @PreAuthorize("@auth.hasAuthority('system:post:query')")
    public Result detail(@PathVariable Long id) {
        SysPost sysPost = sysPostService.getById(id);
        return Result.success(sysPost);
    }

    /**
     * 分页查询
     *
     * @param sysPost  分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "岗位管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:post:query')")
    @GetMapping(value = "/list")
    public Result list(SysPost sysPost
            , @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo
            , @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysPost,检索参数sysRole={},分页参数pageNo={},pageSize={}", sysPost, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysPost> query = getQueryWrapper(sysPost);
            IPage<SysPost> pageList = sysPostService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 删除岗位
     */
    @PreAuthorize("@auth.hasAuthority('system:post:remove')")
    @DeleteMapping("/{postIds}")
    public Result remove(@PathVariable Long[] postIds) {
        return Result.success(sysPostService.deletePostByIds(postIds));
    }

    /**
     * 新增岗位
     */
    @PreAuthorize("@auth.hasAuthority('system:post:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysPost post) {
        if (sysPostService.checkPostNameUnique(post)) {
            return Result.fail("新增岗位'" + post.getPostName() + "'失败，岗位名称已存在");
        } else if (sysPostService.checkPostCodeUnique(post)) {
            return Result.fail("新增岗位'" + post.getPostName() + "'失败，岗位编码已存在");
        }
        post.setCreateBy(jwtService.getUserName());
        post.setCreateTime(LocalDateTime.now());
        return Result.success(sysPostService.save(post));
    }

    /**
     * 修改岗位
     */
    @PreAuthorize("@auth.hasAuthority('system:post:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysPost post) {
        if (sysPostService.checkPostNameUnique(post)) {
            return Result.fail("修改岗位'" + post.getPostName() + "'失败，岗位名称已存在");
        } else if (sysPostService.checkPostCodeUnique(post)) {
            return Result.fail("修改岗位'" + post.getPostName() + "'失败，岗位编码已存在");
        }
        post.setUpdateBy(jwtService.getUserName());
        post.setUpdateTime(LocalDateTime.now());
        return Result.success(sysPostService.updateById(post));
    }

    /**
     * 导出岗位
     */
    @OperateLog(title = "岗位管理", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:post:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysPost sysPost) throws Exception {
        LambdaQueryWrapper<SysPost> query = getQueryWrapper(sysPost);
        List<SysPost> list = sysPostService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("岗位", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysPost.class).sheet("岗位").doWrite(list);
    }

    private LambdaQueryWrapper<SysPost> getQueryWrapper(SysPost sysPost) {
        LambdaQueryWrapper<SysPost> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysPost.getPostName())) {
            query.like(SysPost::getPostName, sysPost.getPostName());
        }
        if (StrUtil.isNotEmpty(sysPost.getPostCode())) {
            query.like(SysPost::getPostCode, sysPost.getPostCode());
        }
        if (StrUtil.isNotEmpty(sysPost.getStatus())) {
            query.eq(SysPost::getStatus, sysPost.getStatus());
        }
        return query;
    }

}
