package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.system.domain.SysOperLog;
import cn.morethank.open.admin.system.service.SysOperLogService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.util.List;

/**
 * 操作日志 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/operlog")
public class SysOperLogController {

    private final SysOperLogService sysOperLogService;

    /**
    * 详情页
    */
    @RequestMapping("{id}")
    public Result detail(@PathVariable Integer id) {
        SysOperLog sysOperLog = sysOperLogService.getById(id);
        return Result.success(sysOperLog);
    }

    /**
    * 删除接口
    */
    @DeleteMapping("{id}")
    public Result delete(@PathVariable Integer id) {
        Boolean deleted = sysOperLogService.removeById(id);
        return Result.success(deleted);
    }

    /**
     * 分页查询
     *
     * @param sysOperLog    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @GetMapping(value = "/list")
    public Result list(SysOperLog sysOperLog,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysOperLog,检索参数sysOperLog={},分页参数pageNo={},pageSize={}", sysOperLog, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysOperLog> query = getQueryWrapper(sysOperLog);
            IPage<SysOperLog> pageList = sysOperLogService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    @OperateLog(title = "操作日志", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:operlog:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysOperLog operLog) throws Exception {
        LambdaQueryWrapper<SysOperLog> query = getQueryWrapper(operLog);
        List<SysOperLog> list = sysOperLogService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("操作日志", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysOperLog.class).sheet("操作日志").doWrite(list);
    }

    @OperateLog(title = "操作日志", businessType = BusinessType.CLEAN)
    @PreAuthorize("@auth.hasAuthority('system:operlog:remove')")
    @DeleteMapping("/clean")
    public Result clean() {
        sysOperLogService.cleanOperLog();
        return Result.success();
    }

    private LambdaQueryWrapper<SysOperLog> getQueryWrapper(SysOperLog sysOperLog) {
        LambdaQueryWrapper<SysOperLog> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysOperLog.getTitle())) {
            query.like(SysOperLog::getTitle, sysOperLog.getTitle());
        }
        if (sysOperLog.getBusinessType() != null) {
            query.eq(SysOperLog::getBusinessType, sysOperLog.getBusinessType());
        }
        if (sysOperLog.getStatus() != null) {
            query.eq(SysOperLog::getStatus, sysOperLog.getStatus());
        }
        if (StrUtil.isNotEmpty(sysOperLog.getOperName())) {
            query.like(SysOperLog::getOperName, sysOperLog.getOperName());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, sysOperLog.getParams());
        // 排序
        query.orderByDesc(SysOperLog::getOperId);
        return query;
    }
}
