package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.system.domain.SysNotice;
import cn.morethank.open.admin.system.service.SysNoticeService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;

/**
 * 通知公告 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/notice")
public class SysNoticeController {

    private final SysNoticeService sysNoticeService;
    private final JwtService jwtService;

    /**
    * 详情页
    */
    @PreAuthorize("@auth.hasAuthority('system:notice:query')")
    @RequestMapping("{id}")
    public Result detail(@PathVariable Long id) {
        SysNotice sysNotice = sysNoticeService.getById(id);
        return Result.success(sysNotice);
    }

    /**
     * 分页查询
     *
     * @param sysNotice 分页查询筛选条件
     * @param pageNo    页码
     * @param pageSize  每页的数量
     * @return 分页查询结果
     */
    @PreAuthorize("@auth.hasAuthority('system:notice:list')")
    @GetMapping(value = "/list")
    public Result list(SysNotice sysNotice, @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysNotice,检索参数sysNotice={},分页参数pageNo={},pageSize={}", sysNotice, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysNotice> query = getQueryWrapper(sysNotice);
            IPage<SysNotice> pageList = sysNoticeService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 删除参数配置
     */
    @PreAuthorize("@auth.hasAuthority('system:notice:remove')")
    @DeleteMapping("/{noticeIds}")
    public Result remove(@PathVariable Long[] noticeIds) {
        return Result.success(sysNoticeService.deleteNoticeByIds(noticeIds));
    }


    /**
     * 新增通知公告
     */
    @PreAuthorize("@auth.hasAuthority('system:notice:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysNotice notice) {
        notice.setCreateBy(jwtService.getUserName());
        notice.setCreateTime(LocalDateTime.now());
        return Result.success(sysNoticeService.save(notice));
    }

    /**
     * 修改通知公告
     */
    @PreAuthorize("@auth.hasAuthority('system:notice:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysNotice notice) {
        notice.setUpdateBy(jwtService.getUserName());
        notice.setUpdateTime(LocalDateTime.now());
        return Result.success(sysNoticeService.updateById(notice));
    }

    private LambdaQueryWrapper<SysNotice> getQueryWrapper(SysNotice sysNotice) {
        LambdaQueryWrapper<SysNotice> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysNotice.getNoticeTitle())) {
            query.like(SysNotice::getNoticeTitle, sysNotice.getNoticeTitle());
        }
        if (StrUtil.isNotEmpty(sysNotice.getNoticeType())) {
            query.eq(SysNotice::getNoticeType, sysNotice.getNoticeType());
        }
        if (StrUtil.isNotEmpty(sysNotice.getCreateBy())) {
            query.eq(SysNotice::getCreateBy, sysNotice.getCreateBy());
        }
        return query;
    }
}
