package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.domain.UserObject;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysMenu;
import cn.morethank.open.admin.system.service.SysMenuService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 菜单管理 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/menu")
public class SysMenuController {

    private final SysMenuService sysMenuService;

    private final JwtService jwtService;

    /**
    * 详情页
    */
     @PreAuthorize("@auth.hasAuthority('system:menu:query')")
    @RequestMapping("{id}")
    public Result detail(@PathVariable Long id) {
        SysMenu sysMenu = sysMenuService.getById(id);
        return Result.success(sysMenu);
    }

    /**
     * 获取菜单列表, 返回所有符合条件的数据, 不是分页查询
     *
     * @param sysMenu    分页查询筛选条件
     * @return 分页查询结果
     */
    @PreAuthorize("@auth.hasAuthority('system:menu:list')")
    @GetMapping(value = "/list")
    public Result list(SysMenu sysMenu) {
        try {
            List<SysMenu> pageList;
            UserObject sysUser = jwtService.getLoginAccount().getUser();
            if (sysUser.isAdmin()) {
                // 管理员显示所有菜单信息
                LambdaQueryWrapper<SysMenu> query = getQueryWrapper(sysMenu);
                pageList = sysMenuService.list(query);
            } else {
                // 如果不是管理员, 则需要连表查询
                sysMenu.getParams().put("userId", sysUser.getUserId());
                pageList = sysMenuService.selectMenuListByUserId(sysMenu);
            }
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 新增菜单
     */
    @PreAuthorize("@auth.hasAuthority('system:menu:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysMenu menu) {
        if (sysMenuService.checkMenuNameUnique(menu)){
            return Result.fail("新增菜单'" + menu.getMenuName() + "'失败，菜单名称已存在");
        } else if (GlobalConstant.YES_FRAME.equals(menu.getIsFrame()) && !StringUtils.ishttp(menu.getPath())) {
            return Result.fail("新增菜单'" + menu.getMenuName() + "'失败，地址必须以http(s)://开头");
        }

        menu.setCreateBy(jwtService.getUserName());
        menu.setCreateTime(LocalDateTime.now());
        return Result.success(sysMenuService.save(menu));
    }

    /**
     * 修改菜单
     */
    @PreAuthorize("@auth.hasAuthority('system:menu:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysMenu menu) {
        if (sysMenuService.checkMenuNameUnique(menu)) {
            return Result.fail("修改菜单'" + menu.getMenuName() + "'失败，菜单名称已存在");
        } else if (GlobalConstant.YES_FRAME.equals(menu.getIsFrame()) && !StringUtils.ishttp(menu.getPath())) {
            return Result.fail("修改菜单'" + menu.getMenuName() + "'失败，地址必须以http(s)://开头");
        } else if (menu.getMenuId().equals(menu.getParentId())) {
            return Result.fail("修改菜单'" + menu.getMenuName() + "'失败，上级菜单不能选择自己");
        }
        menu.setUpdateBy(jwtService.getUserName());
        menu.setUpdateTime(LocalDateTime.now());
        return Result.success(sysMenuService.updateById(menu));
    }

    /**
     * 删除菜单
     */
    @PreAuthorize("@auth.hasAuthority('system:menu:remove')")
    @DeleteMapping("/{menuId}")
    public Result remove(@PathVariable("menuId") Long menuId) {
        if (sysMenuService.hasChildByMenuId(menuId)) {
            return Result.fail("存在子菜单,不允许删除");
        }
        if (sysMenuService.checkMenuExistRole(menuId)) {
            return Result.fail("菜单已分配,不允许删除");
        }
        return Result.success(sysMenuService.removeById(menuId));
    }

    private LambdaQueryWrapper<SysMenu> getQueryWrapper(SysMenu sysMenu) {
        LambdaQueryWrapper<SysMenu> query = new LambdaQueryWrapper<>();
        // 构造分页查询条件
        if (StrUtil.isNotEmpty(sysMenu.getMenuName())) {
            query.like(SysMenu::getMenuName, sysMenu.getMenuName());
        }
        if (StrUtil.isNotEmpty(sysMenu.getVisible())) {
            query.eq(SysMenu::getVisible, sysMenu.getVisible());
        }
        if (StrUtil.isNotEmpty(sysMenu.getStatus())) {
            query.eq(SysMenu::getStatus, sysMenu.getStatus());
        }
        query.orderByAsc(SysMenu::getParentId, SysMenu::getOrderNum);
        return query;
    }

    /**
     * 获取菜单下拉树列表
     */
    @GetMapping("/treeselect")
    public Result treeselect(SysMenu menu) {
        List<SysMenu> menus = sysMenuService.selectMenuList(menu, jwtService.getLoginAccount().getUser());
        return Result.success(sysMenuService.buildMenuTreeSelect(menus));
    }

    /**
     * 加载对应角色菜单列表树
     */
    @GetMapping(value = "/roleMenuTreeselect/{roleId}")
    public Result roleMenuTreeselect(@PathVariable("roleId") Long roleId) {
        List<SysMenu> menus = sysMenuService.selectMenuList(jwtService.getLoginAccount().getUser());
        Map<String, Object> ajax = new HashMap<>(2);
        ajax.put("checkedKeys", sysMenuService.selectMenuListByRoleId(roleId));
        ajax.put("menus", sysMenuService.buildMenuTreeSelect(menus));
        return Result.success(ajax);
    }
}
