package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.system.domain.SysLoginLog;
import cn.morethank.open.admin.system.service.SysLoginLogService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 登录日志 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/loginlog")
public class SysLoginLogController {

    private final SysLoginLogService sysLoginLogService;

    /**
    * 删除接口
    */
    @OperateLog(title = "登录日志", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:loginlog:remove')")
    @DeleteMapping("{ids}")
    public Result delete(@PathVariable Long[] ids) {
        ArrayList<Long> arrayList = new ArrayList<>(ids.length);
		Collections.addAll(arrayList, ids);
        sysLoginLogService.removeByIds(arrayList);
        return Result.success(ids.length);
    }

    /**
     * 分页查询
     *
     * @param sysLoginLog    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "登录日志", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:loginlog:list')")
    @GetMapping(value = "/list")
    public Result list(SysLoginLog sysLoginLog,
                       @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                       @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysLoginLog,检索参数sysLoginLog={},分页参数pageNo={},pageSize={}", sysLoginLog, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysLoginLog> query = getQueryWrapper(sysLoginLog);
            IPage<SysLoginLog> pageList = sysLoginLogService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    @OperateLog(title = "登录日志", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:loginlog:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysLoginLog sysLoginLog) throws Exception {
        LambdaQueryWrapper<SysLoginLog> query = getQueryWrapper(sysLoginLog);
        List<SysLoginLog> list = sysLoginLogService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("登录日志", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysLoginLog.class).sheet("登录日志").doWrite(list);
    }

    @PreAuthorize("@auth.hasAuthority('system:loginlog:remove')")
    @OperateLog(title = "登录日志", businessType = BusinessType.CLEAN)
    @DeleteMapping("/clean")
    public Result clean() {
        sysLoginLogService.cleanLoginLog();
        return Result.success();
    }

    private LambdaQueryWrapper<SysLoginLog> getQueryWrapper(SysLoginLog sysLoginLog) {
        LambdaQueryWrapper<SysLoginLog> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysLoginLog.getIpaddr())) {
            query.like(SysLoginLog::getIpaddr, sysLoginLog.getIpaddr());
        }
        if (StrUtil.isNotEmpty(sysLoginLog.getStatus())) {
            query.eq(SysLoginLog::getStatus, sysLoginLog.getStatus());
        }
        if (StrUtil.isNotEmpty(sysLoginLog.getUserName())) {
            query.eq(SysLoginLog::getUserName, sysLoginLog.getUserName());
        }
        // 添加创建时间的条件
        QueryWrapperUtil.loginTimeCondition(query, sysLoginLog.getParams());
        query.orderByDesc(SysLoginLog::getInfoId);
        return query;
    }
}
