package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.system.domain.SysDictType;
import cn.morethank.open.admin.system.service.SysDictTypeService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 字典数据 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/dict/type")
public class SysDictTypeController {

    private final SysDictTypeService sysDictTypeService;
    private final JwtService jwtService;

    /**
    * 详情页
    */
    @PreAuthorize("@auth.hasAuthority('system:dict:query')")
    @RequestMapping("{id}")
    public Result detail(@PathVariable Long id) {
        SysDictType sysDictType = sysDictTypeService.getById(id);
        return Result.success(sysDictType);
    }

    /**
     * 分页查询
     *
     * @param sysDictType 分页查询筛选条件
     * @param pageNo      页码
     * @param pageSize    每页的数量
     * @return 分页查询结果
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:list')")
    @GetMapping(value = "/list")
    public Result list(SysDictType sysDictType,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysDictType,检索参数sysDictType={},分页参数pageNo={},pageSize={}", sysDictType, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysDictType> query = getQueryWrapper(sysDictType);
            IPage<SysDictType> pageList = sysDictTypeService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 删除字典类型
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:remove')")
    @DeleteMapping("/{dictIds}")
    public Result remove(@PathVariable Long[] dictIds) {
        int count = sysDictTypeService.deleteDictTypeByIds(dictIds);
        return Result.success(count);
    }

    /**
     * 新增字典类型
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysDictType dict) {
        if (sysDictTypeService.checkDictTypeUnique(dict)) {
            return Result.fail("新增字典'" + dict.getDictName() + "'失败，字典类型已存在");
        }
        dict.setCreateBy(jwtService.getUserName());
        dict.setCreateTime(LocalDateTime.now());
        return Result.success(sysDictTypeService.save(dict));
    }

    /**
     * 修改字典类型
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysDictType dict) {
        if (sysDictTypeService.checkDictTypeUnique(dict)) {
            return Result.fail("修改字典'" + dict.getDictName() + "'失败，字典类型已存在");
        }
        dict.setUpdateBy(jwtService.getUserName());
        dict.setUpdateTime(LocalDateTime.now());
        return Result.success(sysDictTypeService.updateById(dict));
    }

    /**
     * 刷新字典缓存
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:remove')")
    @OperateLog(title = "字典类型", businessType = BusinessType.REFRESH)
    @DeleteMapping("/refreshCache")
    public Result refreshCache() {
        sysDictTypeService.resetDictCache();
        return Result.success();
    }

    /**
     * 获取字典选择框列表
     */
    @GetMapping("/optionselect")
    public Result optionselect() {
        List<SysDictType> dictTypes = sysDictTypeService.list();
        return Result.success(dictTypes);
    }


    @OperateLog(title = "字典类型", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:dict:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysDictType sysDictType) throws Exception {
        LambdaQueryWrapper<SysDictType> query = getQueryWrapper(sysDictType);
        List<SysDictType> list = sysDictTypeService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("岗位", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysDictType.class).sheet("岗位").doWrite(list);
    }

    private LambdaQueryWrapper<SysDictType> getQueryWrapper(SysDictType sysDictType) {
        LambdaQueryWrapper<SysDictType> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysDictType.getDictName())) {
            query.like(SysDictType::getDictName, sysDictType.getDictName());
        }
        if (StrUtil.isNotEmpty(sysDictType.getStatus())) {
            query.eq(SysDictType::getStatus, sysDictType.getStatus());
        }
        if (StrUtil.isNotEmpty(sysDictType.getDictType())) {
            query.like(SysDictType::getDictType, sysDictType.getDictType());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, sysDictType.getParams());
        return query;
    }
}
