package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysDictData;
import cn.morethank.open.admin.system.service.SysDictDataService;
import cn.morethank.open.admin.system.service.SysDictTypeService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 字典管理 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/dict/data")
public class SysDictDataController {

    private final SysDictDataService sysDictDataService;
    private final SysDictTypeService sysDictTypeService;
    private final JwtService jwtService;

    /**
    * 详情页
    */
    @PreAuthorize("@auth.hasAuthority('system:dict:query')")
    @RequestMapping("{id}")
    public Result<SysDictData> detail(@PathVariable Long id) {
        SysDictData sysDictData = sysDictDataService.getById(id);
        return Result.success(sysDictData);
    }

    /**
     * 分页查询
     *
     * @param sysDictData    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:list')")
    @GetMapping(value = "/list")
    public Result<IPage<SysDictData>> list(SysDictData sysDictData,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysDictData,检索参数sysDictData={},分页参数pageNo={},pageSize={}", sysDictData, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysDictData> query = getQueryWrapper(sysDictData);
            IPage<SysDictData> pageList = sysDictDataService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 根据字典类型查询字典数据信息
     */
    @GetMapping(value = "/type/{dictType}")
    public Result<List<SysDictData>> dictType(@PathVariable String dictType) {
        List<SysDictData> data = sysDictTypeService.selectDictDataByType(dictType);
        if (StringUtils.isNull(data)) {
            data = new ArrayList<>();
        }
        return Result.success(data);
    }

    /**
     * 删除字典类型
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:remove')")
    @DeleteMapping("/{dictCodes}")
    public Result remove(@PathVariable Long[] dictCodes) {
        sysDictDataService.deleteDictDataByIds(dictCodes);
        return Result.success();
    }

    /**
     * 新增字典类型
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysDictData dict) {
        dict.setCreateBy(jwtService.getUserName());
        dict.setCreateTime(LocalDateTime.now());
        return Result.success(sysDictDataService.save(dict));
    }

    /**
     * 修改字典类型
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysDictData dict) {
        dict.setUpdateBy(jwtService.getUserName());
        dict.setUpdateTime(LocalDateTime.now());
        return Result.success(sysDictDataService.updateById(dict));
    }

    /**
     * 导出数据字典
     * @param response
     * @param dictData 参数
     */
    @PreAuthorize("@auth.hasAuthority('system:dict:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysDictData dictData) throws Exception {
        LambdaQueryWrapper<SysDictData> query = getQueryWrapper(dictData);
        List<SysDictData> list = sysDictDataService.list(query);

        response.setContentType("application/vnd.ms-excel");
        response.setCharacterEncoding("utf-8");
        response.setHeader("Content-disposition", "attachment;filename=post.xlsx");
        EasyExcel.write(response.getOutputStream(), SysDictData.class).sheet("字典数据").doWrite(list);
    }
    
    private LambdaQueryWrapper<SysDictData> getQueryWrapper(SysDictData dict) {
        LambdaQueryWrapper<SysDictData> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(dict.getDictType())) {
            query.eq(SysDictData::getDictType, dict.getDictType());
        }
        if (StrUtil.isNotEmpty(dict.getDictLabel())) {
            query.like(SysDictData::getDictLabel, dict.getDictLabel());
        }
        if (StrUtil.isNotEmpty(dict.getStatus())) {
            query.eq(SysDictData::getStatus, dict.getStatus());
        }
        query.orderByAsc(SysDictData::getDictSort);
        return query;
    }
}
