package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysDept;
import cn.morethank.open.admin.system.service.SysDeptService;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.List;

/**
 * 部门管理 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/dept")
public class SysDeptController {

    private final SysDeptService sysDeptService;
    private final JwtService jwtService;

    /**
    * 详情页
    */
    @PreAuthorize("@auth.hasAuthority('system:dept:query')")
    @RequestMapping("{id}")
    public Result detail(@PathVariable Long id) {
        SysDept sysDept = sysDeptService.getById(id);
        return Result.success(sysDept);
    }

    /**
     * 获取部门列表
     */
    @PreAuthorize("@auth.hasAuthority('system:dept:list')")
    @GetMapping("/list")
    public Result list(SysDept dept) {
        LambdaQueryWrapper<SysDept> query = getQueryWrapper(dept);
        List<SysDept> depts = sysDeptService.list(query);
        return Result.success(depts);
    }

    /**
     * 查询部门列表（排除节点）
     */
    @PreAuthorize("@auth.hasAuthority('system:dept:list')")
    @GetMapping("/list/exclude/{deptId}")
    public Result excludeChild(@PathVariable(value = "deptId", required = false) Long deptId) {
        LambdaQueryWrapper<SysDept> query = getQueryWrapper(new SysDept());
        List<SysDept> depts = sysDeptService.list(query);
        depts.removeIf(d -> d.getDeptId().intValue() == deptId || ArrayUtils.contains(StringUtils.split(d.getAncestors(), ","), deptId + ""));
        return Result.success(depts);
    }

    /**
     * 删除部门
     */
    @PreAuthorize("@auth.hasAuthority('system:dept:remove')")
    @DeleteMapping("/{deptId}")
    public Result remove(@PathVariable Long deptId) {
        if (sysDeptService.hasChildByDeptId(deptId)) {
            return Result.fail("存在下级部门,不允许删除");
        }

        if (sysDeptService.checkDeptExistUser(deptId)) {
            return Result.fail("部门存在用户,不允许删除");
        }
        return Result.success(sysDeptService.deleteDeptById(deptId));
    }

    /**
     * 新增部门
     */
    @PreAuthorize("@auth.hasAuthority('system:dept:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysDept dept) {
        if (sysDeptService.checkDeptNameUnique(dept)) {
            return Result.fail("新增部门'" + dept.getDeptName() + "'失败，部门名称已存在");
        }
        dept.setCreateBy(jwtService.getUserName());
        dept.setCreateTime(LocalDateTime.now());
        return Result.success(sysDeptService.save(dept));
    }

    /**
     * 修改部门
     */
    @PreAuthorize("@auth.hasAuthority('system:dept:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysDept dept) {
        Long deptId = dept.getDeptId();
        if (sysDeptService.checkDeptNameUnique(dept)) {
            return Result.fail("修改部门'" + dept.getDeptName() + "'失败，部门名称已存在");
        } else if (dept.getParentId().equals(deptId)) {
            return Result.fail("修改部门'" + dept.getDeptName() + "'失败，上级部门不能是自己");
        } else if (StringUtils.equals(GlobalConstant.DISABLED, dept.getStatus())
                && sysDeptService.selectNormalChildrenDeptById(deptId) > 0) {
            return Result.fail("该部门包含未停用的子部门！");
        }
        dept.setUpdateBy(jwtService.getUserName());
        dept.setUpdateTime(LocalDateTime.now());
        return Result.success(sysDeptService.updateById(dept));
    }

    private LambdaQueryWrapper<SysDept> getQueryWrapper(SysDept dept) {
        LambdaQueryWrapper<SysDept> query = new LambdaQueryWrapper<>();
        query.eq(SysDept::getDelFlag, GlobalConstant.UNDELETED);
        if (dept.getDeptId() != null && dept.getDeptId() != GlobalConstant.ZERO) {
            query.eq(SysDept::getDeptId, dept.getDeptId());
        }
        if (dept.getParentId() != null && dept.getParentId() != GlobalConstant.ZERO) {
            query.eq(SysDept::getParentId, dept.getParentId());
        }
        if (StrUtil.isNotEmpty(dept.getDeptName())) {
            query.like(SysDept::getDeptName, dept.getDeptName());
        }
        if (StrUtil.isNotEmpty(dept.getStatus())) {
            query.eq(SysDept::getStatus, dept.getStatus());
        }
        query.orderByAsc(SysDept::getParentId, SysDept::getOrderNum);
        return query;
    }
}
