package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.service.DictService;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.system.domain.SysConfig;
import cn.morethank.open.admin.system.service.SysConfigService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 参数配置 前端控制器
 *
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/config")
public class SysConfigController {

    private final SysConfigService sysConfigService;
    private final JwtService jwtService;
    private final DictService dictService;

    /**
    * 详情页
    */
    @PreAuthorize("@auth.hasAuthority('system:config:query')")
    @RequestMapping("{id}")
    public Result detail(@PathVariable Long id) {
        SysConfig sysConfig = sysConfigService.getById(id);
        return Result.success(sysConfig);
    }

    /**
     * 分页查询
     *
     * @param sysConfig 分页查询筛选条件
     * @param pageNo    页码
     * @param pageSize  每页的数量
     * @return 分页查询结果
     */
    @PreAuthorize("@auth.hasAuthority('system:config:list')")
    @GetMapping(value = "/list")
    public Result list(SysConfig sysConfig, @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                       @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询sysConfig,检索参数sysConfig={},分页参数pageNo={},pageSize={}", sysConfig, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<SysConfig> query = getQueryWrapper(sysConfig);
            IPage<SysConfig> pageList = sysConfigService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 删除参数配置
     */
    @PreAuthorize("@auth.hasAuthority('system:config:remove')")
    @DeleteMapping("/{configIds}")
    public Result remove(@PathVariable Long[] configIds) {
        return Result.success(sysConfigService.deleteConfigByIds(configIds));
    }

    /**
     * 刷新参数缓存
     */
    @PreAuthorize("@auth.hasAuthority('system:config:remove')")
    @DeleteMapping("/refreshCache")
    public Result refreshCache() {
        sysConfigService.resetConfigCache();
        return Result.success();
    }

    /**
     * 新增参数配置
     */
    @PreAuthorize("@auth.hasAuthority('system:config:add')")
    @PostMapping
    public Result add(@Validated @RequestBody SysConfig config) {
        if (sysConfigService.checkConfigKeyUnique(config)) {
            return Result.fail("新增参数'" + config.getConfigName() + "'失败，参数键名已存在");
        }
        config.setCreateBy(jwtService.getUserName());
        config.setCreateTime(LocalDateTime.now());
        return Result.success(sysConfigService.save(config));
    }

    /**
     * 修改参数配置
     */
    @PreAuthorize("@auth.hasAuthority('system:config:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody SysConfig config) {
        if (sysConfigService.checkConfigKeyUnique(config)) {
            return Result.fail("修改参数'" + config.getConfigName() + "'失败，参数键名已存在");
        }
        config.setUpdateBy(jwtService.getUserName());
        config.setUpdateTime(LocalDateTime.now());
        return Result.success(sysConfigService.updateById(config));
    }

    /**
     * 导出参数配置
     */
    @OperateLog(title = "参数配置", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:config:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, SysConfig sysConfig) throws Exception {
        LambdaQueryWrapper<SysConfig> query = getQueryWrapper(sysConfig);
        List<SysConfig> list = sysConfigService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("参数配置", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), SysConfig.class).sheet("参数配置").doWrite(list);
    }

    /**
     * 根据参数键名查询参数值
     */
    @GetMapping(value = "/configKey/{configKey}")
    public Result getConfigKey(@PathVariable String configKey) {
        return Result.success(sysConfigService.selectConfigByKey(configKey));
    }

    private LambdaQueryWrapper<SysConfig> getQueryWrapper(SysConfig sysConfig) {
        LambdaQueryWrapper<SysConfig> query = new LambdaQueryWrapper<>();
        if (StrUtil.isNotEmpty(sysConfig.getConfigName())) {
            query.like(SysConfig::getConfigName, sysConfig.getConfigName());
        }
        if (StrUtil.isNotEmpty(sysConfig.getConfigType())) {
            query.eq(SysConfig::getConfigType, sysConfig.getConfigType());
        }
        if (StrUtil.isNotEmpty(sysConfig.getConfigKey())) {
            query.like(SysConfig::getConfigKey, sysConfig.getConfigKey());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, sysConfig.getParams());
        return query;
    }
}
