package cn.morethank.open.admin.system.controller;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.*;
import cn.morethank.open.admin.system.service.ApiProjectService;
import cn.morethank.open.admin.system.domain.ApiProject;

/**
 * API项目 前端控制器
 *
 * @author morethank
 * @since 2023-02-16 21:35:25
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/api/project")
public class ApiProjectController {

    private final ApiProjectService apiProjectService;
    private final JwtService jwtService;

    /**
     * 详情页
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:project:query')")
    @GetMapping("{id}")
    public Result detail(@PathVariable Integer id) {
        ApiProject apiProject = apiProjectService.getById(id);
        return Result.success(apiProject);
    }

    /**
     * 分页查询
     *
     * @param apiProject    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:project:list')")
    @GetMapping(value = "/list")
    public Result list(ApiProject apiProject,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询apiProject,检索参数apiProject={},分页参数pageNo={},pageSize={}", apiProject, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<ApiProject> query = getQueryWrapper(apiProject);
            IPage<ApiProject> pageList = apiProjectService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 获取项目选择框列表
     */
    @GetMapping("/optionselect")
    public Result optionselect() {
        List<ApiProject> projects = apiProjectService.list();
        return Result.success(projects);
    }

    /**
     * 新增保存
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.INSERT)
    @PreAuthorize("@auth.hasAuthority('system:project:add')")
    @PostMapping
    public Result add(@Validated @RequestBody ApiProject apiProject) {
        if (apiProjectService.checkNameUnique(apiProject)) {
            return Result.fail("新增接口文档项目'" + apiProject.getProjectName() + "'失败，项目已存在");
        }
        String uuid = UUID.randomUUID().toString().replaceAll("-", "");
        apiProject.setApikey(uuid);
        apiProject.setCreateBy(jwtService.getUserName());
        apiProject.setCreateTime(LocalDateTime.now());
        return Result.success(apiProjectService.save(apiProject));
    }

    /**
     * 修改保存
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:project:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody ApiProject apiProject) {
        if (apiProjectService.checkNameUnique(apiProject)) {
            return Result.fail("修改接口文档项目'" + apiProject.getProjectName() + "'失败，项目已存在");
        }
        apiProject.setUpdateBy(jwtService.getUserName());
        apiProject.setUpdateTime(LocalDateTime.now());

        return Result.success(apiProjectService.updateById(apiProject));
    }

    /**
     * 刷新apikey
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.REFRESH)
    @PreAuthorize("@auth.hasAuthority('system:project:edit')")
    @PostMapping("/refreshApikey")
    public Result refreshApikey(@Validated @RequestBody ApiProject apiProject) {
        String uuid = UUID.randomUUID().toString().replaceAll("-", "");
        return Result.success(uuid);
    }

    /**
     * 删除接口
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:project:remove')")
    @DeleteMapping("/{ids}")
    public Result remove(@PathVariable Long[] ids) {
        return Result.success(apiProjectService.deleteByIds(ids));
    }

    /**
     * 导出接口
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:project:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, ApiProject apiProject) throws Exception {
        LambdaQueryWrapper<ApiProject> query = getQueryWrapper(apiProject);
        List<ApiProject> list = apiProjectService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("接口文档", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), ApiProject.class).sheet("接口文档").doWrite(list);
    }

    private LambdaQueryWrapper<ApiProject> getQueryWrapper(ApiProject apiProject) {
        LambdaQueryWrapper<ApiProject> query = new LambdaQueryWrapper<>();
        // 添加条件
        if (StrUtil.isNotEmpty(apiProject.getProjectName())) {
            query.like(ApiProject::getProjectName, apiProject.getProjectName());
        }
        if (StrUtil.isNotEmpty(apiProject.getStatus())) {
            query.eq(ApiProject::getStatus, apiProject.getStatus());
        }
        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, apiProject.getParams());
        return query;
    }
}
