package cn.morethank.open.admin.system.controller;

import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.system.domain.ApiInterface;
import cn.morethank.open.admin.system.domain.ApiModule;
import cn.morethank.open.admin.system.service.ApiInterfaceService;
import cn.morethank.open.admin.system.service.ApiModuleService;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.List;

/**
 * API模块管理 前端控制器
 *
 * @author morethank
 * @since 2023-02-16 21:35:25
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/api/module")
public class ApiModuleController {

    private final ApiModuleService apiModuleService;
    private final ApiInterfaceService apiInterfaceService;
    private final JwtService jwtService;

    /**
     * 详情页
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:module:query')")
    @GetMapping("/{id}")
    public Result detail(@PathVariable Integer id) {
        ApiModule apiModule = apiModuleService.getById(id);
        return Result.success(apiModule);
    }

    /**
     * 分页查询
     *
     * @param apiModule    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:module:list')")
    @GetMapping(value = "/list")
    public Result list(ApiModule apiModule,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询apiModule,检索参数apiModule={},分页参数pageNo={},pageSize={}", apiModule, pageNo, pageSize);
        try {
            // 构造分页查询条件
            LambdaQueryWrapper<ApiModule> query = getQueryWrapper(apiModule);
            IPage<ApiModule> pageList = apiModuleService.selectListPage(new Page<>(pageNo, pageSize), query);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 新增保存
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.INSERT)
    @PreAuthorize("@auth.hasAuthority('system:module:add')")
    @PostMapping
    public Result add(@Validated @RequestBody ApiModule apiModule) {
        if (apiModuleService.checkNameUnique(apiModule)) {
            return Result.fail("新增接口文档'" + apiModule.getModuleName() + "'失败，接口文档已存在");
        }
        apiModule.setCreateBy(jwtService.getUserName());
        apiModule.setCreateTime(LocalDateTime.now());
        return Result.success(apiModuleService.save(apiModule));
    }

    /**
     * 修改保存
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:module:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody ApiModule apiModule) {
        if (apiModuleService.checkNameUnique(apiModule)) {
            return Result.fail("修改接口文档'" + apiModule.getModuleName() + "'失败，接口文档已存在");
        }
        apiModule.setUpdateBy(jwtService.getUserName());
        apiModule.setUpdateTime(LocalDateTime.now());

        return Result.success(apiModuleService.updateById(apiModule));
    }

    /**
     * 删除接口
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:module:remove')")
    @DeleteMapping("/{id}")
    public Result remove(@PathVariable Long id) {
        LambdaQueryWrapper<ApiInterface> query = new LambdaQueryWrapper<>();
        query.eq(ApiInterface::getModuleId, id);
        long interfaceCount = apiInterfaceService.count(query);
        if (interfaceCount > 0) {
            return Result.fail("删除模块失败，该模块下存在接口不能被删除。");
        }
        return Result.success(apiModuleService.removeById(id));
    }

    /**
     * 导出接口
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:module:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, ApiModule apiModule) throws Exception {
        LambdaQueryWrapper<ApiModule> query = getQueryWrapper(apiModule);
        List<ApiModule> list = apiModuleService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("接口文档", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), ApiModule.class).sheet("接口文档").doWrite(list);
    }

    /**
     * 获取部门树列表
     */
    @OperateLog(title = "用户管理", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:module:list')")
    @GetMapping("/moduleTree/{projectId}")
    public Result moduleTree(@PathVariable Long projectId) {
        return Result.success(apiModuleService.selectModuleTreeList(projectId));
    }

    private LambdaQueryWrapper<ApiModule> getQueryWrapper(ApiModule apiModule) {
        LambdaQueryWrapper<ApiModule> query = new LambdaQueryWrapper<>();
        // TODO 添加条件
        /*if (StrUtil.isNotEmpty(apiModule.getName())) {
            query.like(ApiModule::getName, apiModule.getName());
        }*/
        return query;
    }
}
