package cn.morethank.open.admin.system.controller;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.annotation.OperateLog;
import cn.morethank.open.admin.common.domain.BusinessType;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.JwtService;
import cn.morethank.open.admin.common.util.QueryWrapperUtil;
import cn.morethank.open.admin.common.util.RequestUtil;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.ApiInterface;
import cn.morethank.open.admin.system.service.ApiInterfaceService;
import cn.morethank.open.admin.system.service.ApiModuleService;
import cn.morethank.open.admin.system.service.ApiProjectService;
import com.alibaba.excel.EasyExcel;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.plugins.pagination.PageDTO;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * API接口文档 前端控制器
 *
 * @author morethank
 * @since 2023-02-20 09:19:22
 */
@Slf4j
@AllArgsConstructor
@RestController
@RequestMapping("/system/api/interface")
public class ApiInterfaceController {

    private final ApiInterfaceService apiInterfaceService;
    private final ApiProjectService apiProjectService;
    private final ApiModuleService apiModuleService;
    private final JwtService jwtService;

    /**
    * 详情页, 编辑页面需要调用详情接口
    */
    @OperateLog(title = "接口文档", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:interface:query')")
    @GetMapping(value = { "/{projectId}/", "/{projectId}/{id}" })
    public Result getInterface(@PathVariable Long projectId, @PathVariable(value = "id", required = false) Long id) {
        Map<String, Object> dataMap = new HashMap<>(2);
        dataMap.put("modules", apiModuleService.selectModuleTreeList(projectId));
        if (StringUtils.isNotNull(id)) {
            ApiInterface apiInterface = apiInterfaceService.getById(id);
            String requestParameter = apiInterface.getRequestParameter();
            if(requestParameter == null) {
                apiInterface.setRequestParamArray(getJSONArray());
            } else {
                try {
                    JSONArray array = JSONArray.parseArray(requestParameter);
                    apiInterface.setRequestParamArray(array);
                } catch (Exception e) {
                    log.warn(e.getMessage(), e);
                }
            }

            String requestBody = apiInterface.getRequestBody();
            if(requestBody == null) {
                apiInterface.setRequestBodyArray(getJSONArray());
            } else {
                try {
                    JSONArray array = JSONArray.parseArray(requestBody);
                    apiInterface.setRequestBodyArray(array);
                } catch (Exception e) {
                    log.warn(e.getMessage(), e);
                }
            }

            String responseData = apiInterface.getResponseData();
            if(responseData == null) {
                apiInterface.setResponseDataJson(new JSONObject());
            } else {
                try {
                    JSONObject object = JSONObject.parseObject(responseData);
                    apiInterface.setResponseDataJson(object);
                } catch (Exception e) {
                    apiInterface.setResponseDataJson(new JSONObject());
                    log.warn(e.getMessage(), e);
                }
            }
            dataMap.put("interface", apiInterface);
        }
        return Result.success(dataMap);
    }

    private JSONArray getJSONArray() {
        JSONArray array = new JSONArray();
        JSONObject jsonObject = new JSONObject();
        array.add(jsonObject);
        return array;
    }

    /**
    * 详情页
    */
    @OperateLog(title = "接口文档", businessType = BusinessType.DETAIL)
    @PreAuthorize("@auth.hasAuthority('system:interface:query')")
    @GetMapping(value = {"/detail" })
    public Result<ApiInterface> detail(Long id) {
        ApiInterface apiInterface = apiInterfaceService.getDetailById(id);
        updateArray(apiInterface);
        return Result.success(apiInterface);
    }

    /**
     * 分页查询
     *
     * @param apiInterface    分页查询筛选条件
     * @param pageNo   页码
     * @param pageSize 每页的数量
     * @return 分页查询结果
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.LIST)
    @PreAuthorize("@auth.hasAuthority('system:interface:list')")
    @GetMapping(value = "/list")
    public Result<IPage<ApiInterface>> list(ApiInterface apiInterface,
                                 @RequestParam(name = "pageNum", defaultValue = "1") Integer pageNo,
                                 @RequestParam(name = "pageSize", defaultValue = "10") Integer pageSize) {
        log.info("分页查询apiInterface,检索参数apiInterface={},分页参数pageNo={},pageSize={}", apiInterface, pageNo, pageSize);
        try {
            Page<ApiInterface> page = PageDTO.of(pageNo, pageSize);
            IPage<ApiInterface> pageList = apiInterfaceService.selectListPage(page, apiInterface);
            log.info("返回查询结果:{}", pageList);
            return Result.success(pageList);
        } catch (Exception e) {
            log.error("分页查询异常", e);
            return null;
        }
    }

    /**
     * 新增保存
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.INSERT)
    @PreAuthorize("@auth.hasAuthority('system:interface:add')")
    @PostMapping
    public Result add(@Validated @RequestBody JSONObject jsonObject) {
        /*if (apiInterfaceService.checkNameUnique(apiInterface)) {
            return Result.fail("新增接口文档'" + apiInterface.getName() + "'失败，接口文档已存在");
        }*/
        ApiInterface apiInterface = getApiInterface(jsonObject);
        updateRequest(apiInterface);
        apiInterface.setCreateBy(jwtService.getUserName());
        apiInterface.setCreateTime(LocalDateTime.now());
        return Result.success(apiInterfaceService.save(apiInterface));
    }

    /**
     * 修改保存
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.UPDATE)
    @PreAuthorize("@auth.hasAuthority('system:interface:edit')")
    @PutMapping
    public Result edit(@Validated @RequestBody JSONObject jsonObject) {
        /*if (apiInterfaceService.checkNameUnique(apiInterface)) {
            return Result.fail("修改接口文档'" + apiInterface.getName() + "'失败，接口文档已存在");
        }*/
        ApiInterface apiInterface = getApiInterface(jsonObject);
        apiInterface.setUpdateBy(jwtService.getUserName());
        apiInterface.setUpdateTime(LocalDateTime.now());

        JSONObject object = JSONObject.parseObject(apiInterface.getResponseData());
        log.info(object.toString());
        return Result.success(apiInterfaceService.updateById(apiInterface));
    }

    private ApiInterface getApiInterface(JSONObject jsonObject) {
        JSONObject obj = null;
        try {
            String jsonStr = jsonObject.getString("responseDataJson");
            obj = JSONObject.parseObject(jsonStr);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        jsonObject.remove("responseDataJson");
        jsonObject.remove("responseData");
        String json2 = jsonObject.toString();
        ApiInterface apiInterface = JSONObject.parseObject(json2, ApiInterface.class);
        apiInterface.setResponseData(obj == null ? "{}" : obj.toString());
        updateRequest(apiInterface);
        return apiInterface;
    }

    /**
     * 删除接口
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.DELETE)
    @PreAuthorize("@auth.hasAuthority('system:interface:remove')")
    @DeleteMapping("/{ids}")
    public Result remove(@PathVariable Long[] ids) {
        return Result.success(apiInterfaceService.deleteByIds(ids));
    }

    /**
     * 导出接口
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.EXPORT)
    @PreAuthorize("@auth.hasAuthority('system:interface:export')")
    @PostMapping("/export")
    public void export(HttpServletResponse response, ApiInterface apiInterface) throws Exception {
        LambdaQueryWrapper<ApiInterface> query = getQueryWrapper(apiInterface);
        List<ApiInterface> list = apiInterfaceService.list(query);

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode("接口文档", "UTF-8");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + fileName + ".xlsx");
        EasyExcel.write(response.getOutputStream(), ApiInterface.class).sheet("接口文档").doWrite(list);
    }

    /**
     * 同步接口
     * 即从java代码中解析接口，并同步到接口表中
     * @param apiInterface
     * @return
     */
    @OperateLog(title = "接口文档", businessType = BusinessType.SYNC)
    @PostMapping("sync")
    public Result sync(@RequestBody ApiInterface apiInterface) {
        Long projectId = apiProjectService.getProjectIdByApikey(apiInterface.getApikey());
        if(projectId == null) {
            return Result.fail("apikey对应的项目不存在");
        }
        if(RequestUtil.isEmpty(apiInterface.getPath()) || RequestUtil.isEmpty(apiInterface.getMethod())) {
            return Result.fail("接口路径或请求方式为空，忽略该接口");
        }
        apiInterface.setCreateBy("api");
        apiInterface.setCreateTime(LocalDateTime.now());
        apiInterface.setProjectId(projectId);
        Long moduleId = apiModuleService.getModuleId(apiInterface);
        apiInterface.setModuleId(moduleId);

        LambdaQueryWrapper<ApiInterface> query = new LambdaQueryWrapper<>();
        query.eq(ApiInterface::getProjectId, apiInterface.getProjectId());
        query.eq(ApiInterface::getMethod, apiInterface.getMethod());
        query.eq(ApiInterface::getPath, apiInterface.getPath());
        long count = apiInterfaceService.count(query);
        if(count > 0) {
            return Result.fail("该接口已存在，忽略不做处理");
        }
        apiInterfaceService.save(apiInterface);
        return Result.success();
    }

    private LambdaQueryWrapper<ApiInterface> getQueryWrapper(ApiInterface apiInterface) {
        LambdaQueryWrapper<ApiInterface> query = new LambdaQueryWrapper<>();
        if(apiInterface.getModuleId() != null) {
            query.eq(ApiInterface::getModuleId, apiInterface.getModuleId());
        }
        if (StrUtil.isNotEmpty(apiInterface.getInterfaceName())) {
            query.like(ApiInterface::getInterfaceName, apiInterface.getInterfaceName());
        }
        if (StrUtil.isNotEmpty(apiInterface.getPath())) {
            query.like(ApiInterface::getPath, apiInterface.getPath());
        }
        if (StrUtil.isNotEmpty(apiInterface.getMethod())) {
            query.eq(ApiInterface::getMethod, apiInterface.getMethod());
        }

        // 添加创建时间的条件
        QueryWrapperUtil.createTimeCondition(query, apiInterface.getParams());
        // 排序
        query.orderByDesc(ApiInterface::getCreateTime);
        return query;
    }

    private void updateRequest(ApiInterface apiInterface) {
        JSONArray paramArray = apiInterface.getRequestParamArray();
        if(paramArray != null && paramArray.size() > 0) {
            String param = JSONObject.toJSONString(paramArray);
            apiInterface.setRequestParameter(param);
        }

        JSONArray bodyArray = apiInterface.getRequestBodyArray();
        if(bodyArray != null && bodyArray.size() > 0) {
            String body = JSONObject.toJSONString(bodyArray);
            apiInterface.setRequestBody(body);
        }
    }

    private void updateArray(ApiInterface apiInterface) {
        String parameter = apiInterface.getRequestParameter();
        if(parameter != null && parameter.trim().length() > 0) {
            try {
                JSONArray array = JSONArray.parseArray(parameter);
                if(array.size() == 1 && array.getJSONObject(0).size() == 0 ) {
                    apiInterface.setRequestParamArray(null);
                } else {
                    apiInterface.setRequestParamArray(array);
                }
            } catch (Exception e) {
            }
        }

        String body = apiInterface.getRequestBody();
        if(body != null && body.trim().length() > 0) {
            try {
                JSONArray array = JSONArray.parseArray(body);
                if(array.size() == 1 && array.getJSONObject(0).size() == 0 ) {
                    apiInterface.setRequestBodyArray(null);
                } else {
                    apiInterface.setRequestBodyArray(array);
                }
            } catch (Exception e) {
            }
        }

        String responseData = apiInterface.getResponseData();
        if(responseData != null && responseData.trim().length() > 0) {
            try {
                JSONObject jsonObject = JSONObject.parseObject(responseData);
                apiInterface.setResponseDataJson(jsonObject);
            } catch (Exception e) {
            }
        }
    }
}
