package cn.morethank.open.admin.common.service;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.exception.ServiceException;
import cn.morethank.open.admin.common.domain.LoginAccount;
import cn.morethank.open.admin.common.util.IpUtils;
import cn.morethank.open.admin.common.util.LogUtil;
import cn.morethank.open.admin.common.util.RequestUtil;
import cn.morethank.open.admin.system.domain.SysUser;
import cn.morethank.open.admin.system.service.SysLoginLogService;
import cn.morethank.open.admin.system.service.SysUserService;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDateTime;

/**
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Service
@Slf4j
public class SysLoginService {

    @Resource
    private AuthenticationManager authenticationManager;

    @Resource
    private JwtService jwtService;

    @Resource
    private RedisService redisService;

    @Resource
    private RsaService rsaService;

    @Resource
    private SysUserService sysUserService;

    @Resource
    private SysLoginLogService sysLoginLogService;

    public String login(String username, String password, String code, String uuid) {
        // 用户验证
        Authentication authentication;
        String user = null;
        try {
            // 根据uuid从缓存中获取私钥
            String privateKey = (String) redisService.hget(GlobalConstant.PRIVATE_KEY, uuid);
            // 解密username和password
            user = rsaService.privateDecrypt(username, privateKey);
            String passwd = rsaService.privateDecrypt(password, privateKey);
            UsernamePasswordAuthenticationToken authToken = new UsernamePasswordAuthenticationToken(user, passwd);
            // 该方法会去调用 UserDetailsServiceImpl.loadUserByUsername
            authentication = authenticationManager.authenticate(authToken);
        } catch (Exception e) {
            if (e instanceof BadCredentialsException) {
                LogUtil.saveLoginLog(sysLoginLogService, user, GlobalConstant.LOGIN_FAIL, GlobalConstant.LOGIN_ERROR);
                throw e;
            } else {
                LogUtil.saveLoginLog(sysLoginLogService, user, GlobalConstant.LOGIN_FAIL, e.getMessage());
                throw new ServiceException(e.getMessage());
            }
        }
        // 记录登录日志
        LogUtil.saveLoginLog(sysLoginLogService, user, GlobalConstant.LOGIN_SUCCESS, GlobalConstant.LOGIN_SUCCESS);
        LoginAccount loginUser = (LoginAccount) authentication.getPrincipal();
        // 更新用户登录相关字段的值
        updateUserLogin(loginUser.getUserId());
        // 生成token
        String token = jwtService.generateToken(loginUser);
        return token;
    }

    /**
     * 更新用户登录相关字段的值
     *
     * @param userId 用户ID
     */
    public void updateUserLogin(Long userId) {
        LambdaUpdateWrapper<SysUser> lambdaUpdateWrapper = new LambdaUpdateWrapper<>();
        lambdaUpdateWrapper.eq(SysUser::getUserId, userId)
                .set(SysUser::getLoginIp, IpUtils.getIpAddr(RequestUtil.getRequest()))
                .set(SysUser::getLoginDate, LocalDateTime.now());
        sysUserService.update(null, lambdaUpdateWrapper);
    }
}
