package cn.morethank.open.admin.common.service;

import cn.hutool.core.io.IoUtil;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import org.springframework.stereotype.Component;

import javax.crypto.Cipher;
import java.io.ByteArrayOutputStream;
import java.security.*;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;

/**
 * RSA 加密解密算法类
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Component
public class RsaService {

    private final int DEFAULT_KEY_SIZE = 1024;

    public Map<String, String> getKeys() {
        return getKeys(DEFAULT_KEY_SIZE);
    }

    public Map<String, String> getKeys(int keySize) {
        // 为RSA算法创建一个KeyPairGenerator对象（KeyPairGenerator，密钥对生成器，用于生成公钥和私钥对）
        KeyPairGenerator kpg;
        try {
            kpg = KeyPairGenerator.getInstance(GlobalConstant.RSA_ALGORITHM);
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("No such algorithm-->[" + GlobalConstant.RSA_ALGORITHM + "]");
        }

        // 初始化KeyPairGenerator对象,密钥长度
        kpg.initialize(keySize);
        // 生成密匙对
        KeyPair keyPair = kpg.generateKeyPair();
        //得到公钥
        Key publicKey = keyPair.getPublic();
        // 返回一个publicKey经过二次加密后的字符串
        String publicKeyStr = Base64.getEncoder().encodeToString(publicKey.getEncoded());
        //得到私钥
        Key privateKey = keyPair.getPrivate();
        // 返回一个privateKey经过二次加密后的字符串
        String privateKeyStr = Base64.getEncoder().encodeToString(privateKey.getEncoded());

        Map<String, String> keyPairMap = new HashMap<>(2);
        keyPairMap.put(GlobalConstant.PUBLIC_KEY, publicKeyStr);
        keyPairMap.put(GlobalConstant.PRIVATE_KEY, privateKeyStr);

        return keyPairMap;
    }

    /**
     * 得到公钥
     * @param publicKey 密钥字符串（经过base64编码）
     * @throws Exception
     */
    private RSAPublicKey getPublicKey(String publicKey) throws NoSuchAlgorithmException, InvalidKeySpecException {
        //通过X509编码的Key指令获得公钥对象
        KeyFactory keyFactory = KeyFactory.getInstance(GlobalConstant.RSA_ALGORITHM);
        X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(Base64.getDecoder().decode(publicKey));
        RSAPublicKey key = (RSAPublicKey) keyFactory.generatePublic(x509KeySpec);
        return key;
    }

    /**
     * 得到私钥
     * @param privateKey 密钥字符串（经过base64编码）
     * @throws Exception
     */
    private RSAPrivateKey getPrivateKey(String privateKey) throws NoSuchAlgorithmException, InvalidKeySpecException {
        //通过PKCS#8编码的Key指令获得私钥对象
        KeyFactory keyFactory = KeyFactory.getInstance(GlobalConstant.RSA_ALGORITHM);
        PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(Base64.getDecoder().decode(privateKey));
        RSAPrivateKey key = (RSAPrivateKey) keyFactory.generatePrivate(pkcs8KeySpec);
        return key;
    }

    /**
     * 公钥加密
     * @param data
     * @param publicKeyStr
     * @return 公钥加密后的文本
     */
    public String publicEncrypt(String data, String publicKeyStr){
        try{
            RSAPublicKey publicKey = getPublicKey(publicKeyStr);
            Cipher cipher = Cipher.getInstance(GlobalConstant.RSA_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, publicKey);
            int bitLength = publicKey.getModulus().bitLength();
            byte[] rsaCodec = rsaSplitCodec(cipher, Cipher.ENCRYPT_MODE, data.getBytes(GlobalConstant.UTF8), bitLength);
            return Base64.getEncoder().encodeToString(rsaCodec);
        } catch(Exception e) {
            throw new RuntimeException("加密字符串[" + data + "]时遇到异常", e);
        }
    }

    /**
     * 私钥解密
     * @param data 密文
     * @param privateKeyStr 私钥字符串
     * @return 私钥解密后的文本
     */
    public String privateDecrypt(String data, String privateKeyStr) {
        try {
            RSAPrivateKey privateKey = getPrivateKey(privateKeyStr);
            Cipher cipher = Cipher.getInstance(GlobalConstant.RSA_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, privateKey);
            byte[] decode = Base64.getDecoder().decode(data);
            int bitLength = privateKey.getModulus().bitLength();
            return new String(rsaSplitCodec(cipher, Cipher.DECRYPT_MODE, decode, bitLength), GlobalConstant.UTF8);
        } catch (Exception e) {
            throw new RuntimeException("解密字符串[" + data + "]时遇到异常", e);
        }
    }

    /**
     * 私钥加密
     * @param data
     * @param privateKey
     * @return 私钥加密后的文本
     */
    public String privateEncrypt(String data, RSAPrivateKey privateKey) {
        try {
            Cipher cipher = Cipher.getInstance(GlobalConstant.RSA_ALGORITHM);
            cipher.init(Cipher.ENCRYPT_MODE, privateKey);
            int bitLength = privateKey.getModulus().bitLength();
            byte[] rsaCodec = rsaSplitCodec(cipher, Cipher.ENCRYPT_MODE, data.getBytes(GlobalConstant.UTF8), bitLength);
            return Base64.getEncoder().encodeToString(rsaCodec);
        } catch (Exception e) {
            throw new RuntimeException("加密字符串[" + data + "]时遇到异常", e);
        }
    }

    /**
     * 公钥解密
     * @param data
     * @param publicKey
     * @return 公钥解密后的文本
     */
    public String publicDecrypt(String data, RSAPublicKey publicKey) {
        try {
            Cipher cipher = Cipher.getInstance(GlobalConstant.RSA_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, publicKey);
            byte[] rsaByte = Base64.getDecoder().decode(data);
            int bitLength = publicKey.getModulus().bitLength();
            return new String(rsaSplitCodec(cipher, Cipher.DECRYPT_MODE, rsaByte, bitLength), GlobalConstant.UTF8);
        } catch (Exception e) {
            throw new RuntimeException("解密字符串[" + data + "]时遇到异常", e);
        }
    }

    private byte[] rsaSplitCodec(Cipher cipher, int opmode, byte[] datas, int keySize) {
        int maxBlock = 0;
        if (opmode == Cipher.DECRYPT_MODE) {
            maxBlock = keySize / 8;
        } else {
            maxBlock = keySize / 8 - 11;
        }

        ByteArrayOutputStream out = new ByteArrayOutputStream();
        byte[] resultDatas;
        try {
            int offSet = 0;
            byte[] buff;
            int i = 0;
            while (datas.length > offSet) {
                if (datas.length - offSet > maxBlock) {
                    buff = cipher.doFinal(datas, offSet, maxBlock);
                } else {
                    buff = cipher.doFinal(datas, offSet, datas.length - offSet);
                }
                out.write(buff, 0, buff.length);
                i++;
                offSet = i * maxBlock;
            }
            resultDatas = out.toByteArray();
        } catch (Exception e) {
            throw new RuntimeException("加解密阀值为[" + maxBlock + "]的数据时发生异常", e);
        } finally {
            IoUtil.close(out);
        }
        return resultDatas;
    }

}
