package cn.morethank.open.admin.common.service;

import cn.hutool.core.lang.UUID;
import cn.hutool.http.useragent.UserAgent;
import cn.hutool.http.useragent.UserAgentUtil;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.constant.RedisKeyConstant;
import cn.morethank.open.admin.common.domain.LoginAccount;
import cn.morethank.open.admin.common.util.IpUtils;
import cn.morethank.open.admin.common.util.RequestUtil;
import cn.morethank.open.admin.common.util.StringUtils;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.io.Decoders;
import io.jsonwebtoken.security.Keys;
import lombok.Data;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.security.Key;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Data
@Component
@ConfigurationProperties(prefix = "open.admin.jwt")
public class JwtService {

	private long expire;
	private String secret;
	private String header;
	private long multiple = 4;

    @Resource
    private RedisService redisService;

	/**
	 * 生成jwt
	 * @param loginUser
	 * @return 返回token
	 */
	public String generateToken(LoginAccount loginUser) {
		Date nowDate = new Date();
		Date expireDate = new Date(nowDate.getTime() + 1000 * expire);

		byte[] keyBytes = Decoders.BASE64.decode(secret);
		Key key = Keys.hmacShaKeyFor(keyBytes);

		// 允许同一个账号同时多人登录, 用token区分
		String uuid = UUID.fastUUID().toString();
		loginUser.setToken(uuid);
		setUserAgent(loginUser);
		refreshToken(loginUser);

		// 把刚刚生成的uuid放入到jwt参数中
        Map<String, Object> claims = new HashMap<>(1);
        claims.put(RedisKeyConstant.LOGIN_USER_KEY, uuid);

		return Jwts.builder().setClaims(claims)
				.setHeaderParam("typ", "JWT")
				.setSubject(loginUser.getUsername())
				.setIssuedAt(nowDate)
				.setExpiration(expireDate)
				.signWith(key)
				.compact();
	}

    /**
     * 刷新令牌在redis中的有效期
     *
     * @param loginUser 登录信息
     */
    public void refreshToken(LoginAccount loginUser) {
        loginUser.setLoginTime(System.currentTimeMillis());
        loginUser.setExpireTime(loginUser.getLoginTime() + expire);
        // 根据uuid将loginUser缓存
        String userKey = getTokenKey(loginUser.getToken());
        // TODO 这里并没有刷新相同用户的其他登录账号的缓存
        redisService.set(userKey, loginUser, expire);
    }

    /**
     * 验证令牌有效期，剩余时长不足token有效期的1/4时，自动刷新缓存中后端token的有效期
     *
     * @param loginUser
     */
    public void verifyToken(LoginAccount loginUser) {
        long expireTime = loginUser.getExpireTime();
        long currentTime = System.currentTimeMillis();
        if (expireTime - currentTime <= expire/multiple) {
            refreshToken(loginUser);
        }
    }

    private String getTokenKey(String uuid) {
        return RedisKeyConstant.LOGIN_TOKEN_KEY + uuid;
    }

    /**
     * 解析jwt
     * @param jwt
     * @return
     */
	private Claims getClaimByToken(String jwt) {
		try {
			if(jwt.startsWith(GlobalConstant.BEARER)) {
				jwt = jwt.substring(GlobalConstant.BEARER.length()).trim();
			}
			byte[] keyBytes = Decoders.BASE64.decode(secret);
			Key key = Keys.hmacShaKeyFor(keyBytes);
			return Jwts.parserBuilder().setSigningKey(key).build().parseClaimsJws(jwt).getBody();
		} catch (Exception e) {
			return null;
		}
	}

    /**
     * 设置用户代理信息
     *
     * @param loginUser 登录信息
     */
    private void setUserAgent(LoginAccount loginUser) {
		HttpServletRequest request = RequestUtil.getRequest();
        UserAgent userAgent = UserAgentUtil.parse(request.getHeader(GlobalConstant.USER_AGENT));
        String ip = IpUtils.getIpAddr(request);
        loginUser.setIpaddr(ip);
        loginUser.setLoginLocation(IpUtils.getRealAddressByIP(ip));
        loginUser.setBrowser(userAgent.getBrowser().getName());
        loginUser.setOs(userAgent.getOs().getName());
    }

	public LoginAccount getLoginAccount(HttpServletRequest request) {
        // 获取请求携带的令牌
        String token = request.getHeader(header);
        if (StringUtils.isNotEmpty(token)) {
            try {
                Claims claims = getClaimByToken(token);
				if(claims != null) {
					// 解析对应的权限以及用户信息, 如果uuid没有获取到会返回null
					String uuid = (String) claims.get(RedisKeyConstant.LOGIN_USER_KEY);
					String userKey = getTokenKey(uuid);
					LoginAccount loginAccount = (LoginAccount) redisService.get(userKey);
					return loginAccount;
				}
            } catch (Exception e) {
            }
        }
        return null;
	}

    /**
     * 获取当前用户的登录名
     * @return 返回登录名
     */
    public String getUserName() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        LoginAccount loginAccount = (LoginAccount) authentication.getPrincipal();
        return loginAccount.getUsername();
    }

    public LoginAccount getLoginAccount() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        Object principal = authentication.getPrincipal();
        if (principal != null && principal instanceof LoginAccount) {
            LoginAccount loginAccount = (LoginAccount) authentication.getPrincipal();
            return loginAccount;
        }
        return null;
    }

    public void setLoginAccount(LoginAccount loginAccount) {
        if (StringUtils.isNotNull(loginAccount) && StringUtils.isNotEmpty(loginAccount.getToken())) {
            refreshToken(loginAccount);
        }
    }
}