package cn.morethank.open.admin.common.service;

import cn.morethank.open.admin.common.util.DateUtils;
import cn.morethank.open.admin.common.util.StringUtils;
import cn.morethank.open.admin.system.domain.SysGenColumn;
import cn.morethank.open.admin.system.domain.SysGenTable;
import cn.morethank.open.admin.common.util.GenConstants;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @author morethank
 * @since 2023/3/25 20:08
 */
@Service
public class FreemarkerService {

    /**
     * 项目空间路径
     */
    private static final String PROJECT_PATH = "main/java";

    /**
     * mybatis空间路径
     */
    private static final String MYBATIS_PATH = "main/resources/mapper";

    /**
     * 默认上级菜单，系统工具
     */
    private static final String DEFAULT_PARENT_MENU_ID = "3";

    /**
     * 获取模板信息
     *
     * @return 模板列表
     */
    public List<String> getTemplateList(String tplCategory) {
        List<String> templates = new ArrayList<>();
        templates.add("java/domain.java.ftl");
        templates.add("java/mapper.java.ftl");
        templates.add("java/service.java.ftl");
        templates.add("java/serviceImpl.java.ftl");
        templates.add("java/controller.java.ftl");
        templates.add("xml/mapper.xml.ftl");
        templates.add("sql/sql.ftl");
        templates.add("js/api.js.ftl");
        if (GenConstants.TPL_CRUD.equals(tplCategory)) {
            templates.add("vue/index.vue.ftl");
        } else if (GenConstants.TPL_TREE.equals(tplCategory)) {
            templates.add("vue/index-tree.vue.ftl");
        } else if (GenConstants.TPL_SUB.equals(tplCategory)) {
            templates.add("vue/index.vue.ftl");
            templates.add("java/sub-domain.java.ftl");
        }
        return templates;
    }

    public Map<String, Object> getObjectMap(SysGenTable genTable) {
        Map<String, Object> objectMap = new HashMap<>();

        String moduleName = genTable.getModuleName();
        String businessName = genTable.getBusinessName();
        String packageName = genTable.getPackageName();
        String tplCategory = genTable.getTplCategory();
        String functionName = genTable.getFunctionName();

        objectMap.put("tplCategory", genTable.getTplCategory());
        objectMap.put("tableName", genTable.getTableName());
        objectMap.put("functionName", StringUtils.isNotEmpty(functionName) ? functionName : "【请填写功能名称】");
        objectMap.put("ClassName", genTable.getClassName());
        objectMap.put("moduleName", genTable.getModuleName());
        objectMap.put("BusinessName", StringUtils.capitalize(genTable.getBusinessName()));
        objectMap.put("businessName", genTable.getBusinessName());
        objectMap.put("basePackage", getPackagePrefix(packageName));
        objectMap.put("packageName", packageName);
        objectMap.put("author", genTable.getFunctionAuthor());
        objectMap.put("datetime", DateUtils.getTime());
        objectMap.put("pkColumn", genTable.getPkColumn());
        objectMap.put("importList", getImportList(genTable));
        objectMap.put("permissionPrefix", getPermissionPrefix(moduleName, businessName));
        objectMap.put("columns", genTable.getColumns());
        objectMap.put("table", genTable);
        objectMap.put("dicts", getDicts(genTable));
        setMenuVelocityContext(objectMap, genTable);
        if (GenConstants.TPL_TREE.equals(tplCategory)) {
            setTreeVelocityContext(objectMap, genTable);
        }
        if (GenConstants.TPL_SUB.equals(tplCategory)) {
            setSubVelocityContext(objectMap, genTable);
        }
        return objectMap;
    }


    private void setMenuVelocityContext(Map<String, Object> objectMap, SysGenTable genTable) {
        String options = genTable.getOptions();
        JSONObject paramsObj = JSON.parseObject(options);
        String parentMenuId = getParentMenuId(paramsObj);
        objectMap.put("parentMenuId", parentMenuId);
    }

    private void setTreeVelocityContext(Map<String, Object> objectMap, SysGenTable genTable) {
        String options = genTable.getOptions();
        JSONObject paramsObj = JSON.parseObject(options);
        String treeCode = getTreecode(paramsObj);
        String treeParentCode = getTreeParentCode(paramsObj);
        String treeName = getTreeName(paramsObj);

        objectMap.put("treeCode", treeCode);
        objectMap.put("treeParentCode", treeParentCode);
        objectMap.put("treeName", treeName);
        objectMap.put("expandColumn", getExpandColumn(genTable));
        if (paramsObj.containsKey(GenConstants.TREE_PARENT_CODE)) {
            objectMap.put("tree_parent_code", paramsObj.getString(GenConstants.TREE_PARENT_CODE));
        }
        if (paramsObj.containsKey(GenConstants.TREE_NAME)) {
            objectMap.put("tree_name", paramsObj.getString(GenConstants.TREE_NAME));
        }
    }

    private void setSubVelocityContext(Map<String, Object> objectMap, SysGenTable genTable) {
        SysGenTable subTable = genTable.getSubTable();
        String subTableName = genTable.getSubTableName();
        String subTableFkName = genTable.getSubTableFkName();
        String subClassName = genTable.getSubTable().getClassName();
        String subTableFkClassName = StringUtils.convertToCamelCase(subTableFkName);

        objectMap.put("subTable", subTable);
        objectMap.put("subTableName", subTableName);
        objectMap.put("subTableFkName", subTableFkName);
        objectMap.put("subTableFkClassName", subTableFkClassName);
        objectMap.put("subTableFkclassName", StringUtils.uncapitalize(subTableFkClassName));
        objectMap.put("subClassName", subClassName);
        objectMap.put("subclassName", StringUtils.uncapitalize(subClassName));
        objectMap.put("subImportList", getImportList(genTable.getSubTable()));
    }


    /**
     * 获取文件名
     */
    public String getFileName(String template, SysGenTable genTable) {
        // 文件名称
        String fileName = "";
        // 包路径
        String packageName = genTable.getPackageName();
        // 模块名
        String moduleName = genTable.getModuleName();
        // 大写类名
        String className = genTable.getClassName();
        // 业务名称
        String businessName = genTable.getBusinessName();

        String javaPath = PROJECT_PATH + "/" + StringUtils.replace(packageName, ".", "/");
        String mybatisPath = MYBATIS_PATH + "/" + moduleName;
        String vuePath = "vue";

        if (template.contains("domain.java.ftl")) {
            fileName = StringUtils.format("{}/domain/{}.java", javaPath, className);
        }
        if (template.contains("sub-domain.java.ftl") && StringUtils.equals(GenConstants.TPL_SUB, genTable.getTplCategory())) {
            fileName = StringUtils.format("{}/domain/{}.java", javaPath, genTable.getSubTable().getClassName());
        } else if (template.contains("mapper.java.ftl")) {
            fileName = StringUtils.format("{}/mapper/{}Mapper.java", javaPath, className);
        } else if (template.contains("service.java.ftl")) {
            fileName = StringUtils.format("{}/service/I{}Service.java", javaPath, className);
        } else if (template.contains("serviceImpl.java.ftl")) {
            fileName = StringUtils.format("{}/service/impl/{}ServiceImpl.java", javaPath, className);
        } else if (template.contains("controller.java.ftl")) {
            fileName = StringUtils.format("{}/controller/{}Controller.java", javaPath, className);
        } else if (template.contains("mapper.xml.ftl")) {
            fileName = StringUtils.format("{}/{}Mapper.xml", mybatisPath, className);
        } else if (template.contains("sql.ftl")) {
            fileName = businessName + "Menu.sql";
        } else if (template.contains("api.js.ftl")) {
            fileName = StringUtils.format("{}/api/{}/{}.js", vuePath, moduleName, businessName);
        } else if (template.contains("index.vue.ftl")) {
            fileName = StringUtils.format("{}/views/{}/{}/index.vue", vuePath, moduleName, businessName);
        } else if (template.contains("index-tree.vue.ftl")) {
            fileName = StringUtils.format("{}/views/{}/{}/index.vue", vuePath, moduleName, businessName);
        }
        return fileName;
    }

    /**
     * 获取包前缀
     *
     * @param packageName 包名称
     * @return 包前缀名称
     */
    private String getPackagePrefix(String packageName) {
        int lastIndex = packageName.lastIndexOf(".");
        return StringUtils.substring(packageName, 0, lastIndex);
    }

    /**
     * 根据列类型获取导入包
     *
     * @param genTable 业务表对象
     * @return 返回需要导入的包列表
     */
    private HashSet<String> getImportList(SysGenTable genTable) {
        List<SysGenColumn> columns = genTable.getColumns();
        SysGenTable subGenTable = genTable.getSubTable();
        HashSet<String> importList = new HashSet<String>();
        if (StringUtils.isNotNull(subGenTable)) {
            importList.add("java.util.List");
        }
        for (SysGenColumn column : columns) {
            if (!column.isSuperColumn() && GenConstants.TYPE_DATE.equals(column.getJavaType())) {
                importList.add("java.util.Date");
                importList.add("com.fasterxml.jackson.annotation.JsonFormat");
            } else if (!column.isSuperColumn() && GenConstants.TYPE_BIGDECIMAL.equals(column.getJavaType())) {
                importList.add("java.math.BigDecimal");
            }
        }
        return importList;
    }

    /**
     * 根据列类型获取字典组
     *
     * @param genTable 业务表对象
     * @return 返回字典组
     */
    private String getDicts(SysGenTable genTable) {
        List<SysGenColumn> columns = genTable.getColumns();
        Set<String> dicts = new HashSet<String>();
        addDicts(dicts, columns);
        if (StringUtils.isNotNull(genTable.getSubTable())) {
            List<SysGenColumn> subColumns = genTable.getSubTable().getColumns();
            addDicts(dicts, subColumns);
        }
        return StringUtils.join(dicts, ", ");
    }

    /**
     * 添加字典列表
     *
     * @param dicts   字典列表
     * @param columns 列集合
     */
    private void addDicts(Set<String> dicts, List<SysGenColumn> columns) {
        for (SysGenColumn column : columns) {
            if (!column.isSuperColumn() && StringUtils.isNotEmpty(column.getDictType()) && StringUtils.equalsAny(
                    column.getHtmlType(),
                    new String[]{GenConstants.HTML_SELECT, GenConstants.HTML_RADIO, GenConstants.HTML_CHECKBOX})) {
                dicts.add("'" + column.getDictType() + "'");
            }
        }
    }

    /**
     * 获取权限前缀
     *
     * @param moduleName   模块名称
     * @param businessName 业务名称
     * @return 返回权限前缀
     */
    private String getPermissionPrefix(String moduleName, String businessName) {
        return StringUtils.format("{}:{}", moduleName, businessName);
    }

    /**
     * 获取上级菜单ID字段
     *
     * @param paramsObj 生成其他选项
     * @return 上级菜单ID字段
     */
    private String getParentMenuId(JSONObject paramsObj) {
        if (StringUtils.isNotEmpty(paramsObj) && paramsObj.containsKey(GenConstants.PARENT_MENU_ID)
                && StringUtils.isNotEmpty(paramsObj.getString(GenConstants.PARENT_MENU_ID))) {
            return paramsObj.getString(GenConstants.PARENT_MENU_ID);
        }
        return DEFAULT_PARENT_MENU_ID;
    }

    /**
     * 获取树编码
     *
     * @param paramsObj 生成其他选项
     * @return 树编码
     */
    private String getTreecode(JSONObject paramsObj) {
        if (paramsObj.containsKey(GenConstants.TREE_CODE)) {
            return StringUtils.toCamelCase(paramsObj.getString(GenConstants.TREE_CODE));
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取树父编码
     *
     * @param paramsObj 生成其他选项
     * @return 树父编码
     */
    private String getTreeParentCode(JSONObject paramsObj) {
        if (paramsObj.containsKey(GenConstants.TREE_PARENT_CODE)) {
            return StringUtils.toCamelCase(paramsObj.getString(GenConstants.TREE_PARENT_CODE));
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取树名称
     *
     * @param paramsObj 生成其他选项
     * @return 树名称
     */
    private String getTreeName(JSONObject paramsObj) {
        if (paramsObj.containsKey(GenConstants.TREE_NAME)) {
            return StringUtils.toCamelCase(paramsObj.getString(GenConstants.TREE_NAME));
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取需要在哪一列上面显示展开按钮
     *
     * @param genTable 业务表对象
     * @return 展开按钮列序号
     */
    private int getExpandColumn(SysGenTable genTable) {
        String options = genTable.getOptions();
        JSONObject paramsObj = JSON.parseObject(options);
        String treeName = paramsObj.getString(GenConstants.TREE_NAME);
        int num = 0;
        for (SysGenColumn column : genTable.getColumns()) {
            if (column.isList()) {
                num++;
                String columnName = column.getColumnName();
                if (columnName.equals(treeName)) {
                    break;
                }
            }
        }
        return num;
    }

}
