package cn.morethank.open.admin.common.service;

import cn.hutool.core.util.StrUtil;
import cn.morethank.open.admin.common.domain.LoginAccount;
import cn.morethank.open.admin.common.util.StringUtils;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Set;

/**
 * 授权服务
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Service("auth")
public class AuthorityService {

    /** 所有权限标识 */
    public static final String ALL_PERMISSION = "*:*:*";

    /**
     * 验证用户是否具备某权限
     *
     * @param permission 权限字符串
     * @return 用户是否具备某权限
     */
    public boolean hasAuthority(String permission) {
        if (StrUtil.isBlank(permission)) {
			return false;
		}

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
		if (authentication == null) {
			return false;
		}
//        Collection<? extends GrantedAuthority> authorities = authentication.getAuthorities();
//        boolean pass =  hasAuthority(authorities, permission);
        LoginAccount user = (LoginAccount) authentication.getPrincipal();
        if(user.getUser().isAdmin()) {
            return true;
        }
        return hasAuthority(user.getPermissions(), permission);
    }

    /**
     * 判断是否包含权限
     *
     * @param permissions 权限列表
     * @param permission 权限字符串
     * @return 用户是否具备某权限
     */
    private boolean hasAuthority(Set<String> permissions, String permission) {
        return permissions.contains(ALL_PERMISSION) || permissions.contains(StringUtils.trim(permission));
    }

    /**
     * 判断是否包含权限
     *
     * @param authorities 权限列表
     * @param permission 权限字符串
     * @return 用户是否具备某权限
     */
    private boolean hasAuthority(Collection<? extends GrantedAuthority> authorities, String permission) {
        if(authorities.contains(ALL_PERMISSION)) {
            return true;
        }
        for (GrantedAuthority authority: authorities) {
            if(authority.getAuthority().equals(StringUtils.trim(permission))) {
                return true;
            }
        }
        return false;
    }
}
