package cn.morethank.open.admin.common.inject;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.util.RequestUtil;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Map;

/**
 * 请求封装, 为了解决防注入拦截器处理后Controller层获取不到数据的问题
 * 也对内容进行了xss处理, 如果有需要可以考虑加入配置参数对某些参数/接口不进行处理
 * @author morethank
 * @since 2022/12/17 17:23
 */
public class AntiInjectRequestWrapper extends HttpServletRequestWrapper {

    /**
     * 用于将流保存下来
     */
    private byte[] requestBody;

    private Map<String, String[]> parameterMap;

    public AntiInjectRequestWrapper(HttpServletRequest request, ServletResponse response) throws IOException {
        super(request);
        request.setCharacterEncoding(GlobalConstant.UTF8);
        response.setCharacterEncoding(GlobalConstant.UTF8);

        parameterMap = request.getParameterMap();
        String bodyStr = RequestUtil.getBodyString(request);
        requestBody = bodyStr.getBytes(GlobalConstant.UTF8);
    }

    @Override
    public String[] getParameterValues(String name) {
        if(parameterMap != null && parameterMap.size() > 0) {
            String[] values = parameterMap.get(name);
            if (values != null) {
                int length = values.length;
                String[] escapseValues = new String[length];
                for (int i = 0; i < length; i++) {
                    // 防xss攻击和过滤前后空格
                    escapseValues[i] = RequestUtil.xssFilter(values[i]).trim();
                }
                return escapseValues;
            }
        }
        return null;
    }

    @Override
    public String getParameter(String name) {
        String value = super.getParameter(name);
        if(!RequestUtil.isEmpty(value)) {
            value = RequestUtil.xssFilter(value).trim();
        }
        return value;
    }

    @Override
    public ServletInputStream getInputStream() throws IOException {

        final ByteArrayInputStream bais = new ByteArrayInputStream(requestBody);

        return new ServletInputStream() {

            @Override
            public int read() throws IOException {
                return bais.read();
            }

            @Override
            public int available() throws IOException {
                return requestBody.length;
            }

            @Override
            public boolean isFinished() {
                return false;
            }

            @Override
            public boolean isReady() {
                return false;
            }

            @Override
            public void setReadListener(ReadListener readListener) {
            }
        };
    }

    @Override
    public BufferedReader getReader() throws IOException {
        return new BufferedReader(new InputStreamReader(getInputStream()));
    }
}
