package cn.morethank.open.admin.common.inject;

import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.exception.FailedReqeustException;
import cn.morethank.open.admin.common.util.RequestUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.servlet.HandlerInterceptor;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Enumeration;

/**
 * 防注入拦截器
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@Component
public class AntiInjectInterceptor implements HandlerInterceptor {

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        log.debug("------------AntiInjectInterceptor begin--------------");
        String servletPath = request.getServletPath();
        String url = request.getPathInfo() == null ? servletPath : servletPath + request.getPathInfo();

        // 获取请求所有参数，校验防止SQL注入，防止XSS漏洞
        checkParams(request, url, request.getParameterNames());
        // 请求头参数防注入
        checkParams(request, url, request.getHeaderNames());

        String method = request.getMethod().toUpperCase();
        // 本项目新增操作用POST, 修改操作用PUT
        if (GlobalConstant.POST.equalsIgnoreCase(method) || GlobalConstant.PUT.equalsIgnoreCase(method)) {
            String body = RequestUtil.getBodyString(request);
            if (StringUtils.hasLength(body)) {
                // 执行XSS清理
                log.info("{} - {} XSS处理前参数：{}", method, url, body);
                body = AntiInjectXssUtils.xssPostClean(body);
                log.info("{} - {} XSS处理后参数：{}", method, url, body);
            }

            //  如果存在sql注入,直接拦截请求
            if (body.contains(GlobalConstant.ADMIN_FORBIDDEN)) {
                log.error("{} - [{}：{}] 参数：{}, 包含不允许sql的关键词，请求拒绝", method, url, body);
                throw new FailedReqeustException("包含不允许的非法关键词，请求拒绝");
            }
        }
        return true;
    }

    private void checkParams(HttpServletRequest request, String url, Enumeration<?> params) {
        String paramName = null;
        String paramVale = null;
        while (params.hasMoreElements()) {
            paramName = (String) params.nextElement();
            paramVale = request.getParameter(paramName);
            String oldValue = paramVale;
            log.debug("原请求参数值为：{}", paramVale);
            try {
                paramVale = AntiInjectXssUtils.xssGetClean(paramVale);
            } catch (Exception e) {
                log.error("请求清理xss攻击异常", e);
                log.error("请求【" + url + "】参数中清理xss攻击异常, 请求拒绝");
                throw new FailedReqeustException("invalid xss attack");
            }

            log.debug("修改后参数值为：{}", paramVale);
            //    如果存在sql注入,直接拦截请求
            if (paramVale != null && paramVale.contains(GlobalConstant.ADMIN_FORBIDDEN)) {
                String message = paramName + " invalid content：" + oldValue;
                log.error("请求【" + url + "】参数中包含不允许sql的关键词, 请求拒绝");
                throw new FailedReqeustException(message);
            }
        }
    }
}
