package cn.morethank.open.admin.common.exception;

import cn.morethank.open.admin.common.domain.ErrorCode;
import cn.morethank.open.admin.common.domain.Result;
import io.jsonwebtoken.JwtException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.security.access.AccessDeniedException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestControllerAdvice;

/**
 * @author morethank
 * @since 2022/12/17 17:23
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {

    @ExceptionHandler(value = AccessDeniedException.class)
    public Result handler(AccessDeniedException e) {
        e.printStackTrace();
        log.info("security权限不足：{}", e.getMessage());
        return Result.fail(ErrorCode.USER_ACCESS_ERROR.code(), e.getMessage());
    }

    @ExceptionHandler(value = JwtException.class)
    public Result handler(JwtException e) {
        e.printStackTrace();
        log.info("登录状态已过期：{}", e.getMessage());
        return Result.unauthorized();
    }

    @ResponseStatus(HttpStatus.OK)
    @ExceptionHandler(value = ServiceException.class)
    public Result handler(ServiceException e) {
        log.info("业务异常：{}", e.getMessage());
        return Result.fail(ErrorCode.USER_PWD_ERROR.code(), e.getMessage());
    }

    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    public Result handler(MethodArgumentNotValidException e) {
        log.info("实体校验异常：{}", e.getMessage());
        BindingResult bindingResult = e.getBindingResult();
        ObjectError objectError = bindingResult.getAllErrors().stream().findFirst().get();
        return Result.fail(ErrorCode.ENTITY_VALID_ERROR.code(), objectError.getDefaultMessage());
    }

    @ExceptionHandler(value = IllegalArgumentException.class)
    public Result handler(IllegalArgumentException e) {
        log.error("Assert异常：{}", e.getMessage());
        return Result.fail(ErrorCode.UNKNOWN.code(), e.getMessage());
    }

    @ExceptionHandler(value = RuntimeException.class)
    public Result handler(RuntimeException e) {
        log.error("运行时异常：{}", e);
        return Result.fail(ErrorCode.UNKNOWN.code(), e.getMessage());
    }
}
