package cn.morethank.open.admin.common.domain;

import cn.hutool.core.io.IoUtil;
import cn.morethank.open.admin.common.util.RandomUtil;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.OutputStream;

/**
 * @author morethank
 * @since 2022/12/17 17:23
 */
public class ImageCaptcha {

    private Font font = new Font("Verdana", Font.ITALIC | Font.BOLD, 28);
    /**
     * 验证码随机字符长度
     */
    private int len = 5;
    /**
     * 验证码显示宽带
     */
    private int width = 150;
    /**
     * 验证码显示高度
     */
    private int height = 40;
    /**
     * 随机字符串
     */
    private String chars = null;

    public ImageCaptcha() {
    }

    public ImageCaptcha(int width, int height) {
        this.width = width;
        this.height = height;
    }

    public ImageCaptcha(int width, int height, int len) {
        this(width, height);
        this.len = len;
    }

    public ImageCaptcha(int width, int height, int len, Font font) {
        this(width, height, len);
        this.font = font;
    }

    /**
     * 生成验证码
     *
     * @throws java.io.IOException IO异常
     */
    public void out(OutputStream out) {
        graphicsImage(alphas(), out);
    }

    /**
     * 画随机码图
     *
     * @param strs 文本
     * @param out  输出流
     */
    private boolean graphicsImage(char[] strs, OutputStream out) {
        boolean ok = false;
        try {
            BufferedImage bi = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = (Graphics2D) bi.getGraphics();
            AlphaComposite ac3;
            Color color;
            int len = strs.length;
            g.setColor(Color.WHITE);
            g.fillRect(0, 0, width, height);
            // 随机画干扰的蛋蛋
            for (int i = 0; i < 15; i++) {
                color = color(150, 250);
                g.setColor(color);
                // 画蛋蛋，有蛋的生活才精彩
                g.drawOval(RandomUtil.num(width), RandomUtil.num(height), 5 + RandomUtil.num(10), 5 + RandomUtil.num(10));
                color = null;
            }
            g.setFont(font);
            int h = height - ((height - font.getSize()) >> 1), w = width / len, size = w - font.getSize() + 1;
            /* 画字符串 */
            for (int i = 0; i < len; i++) {
                // 指定透明度
                ac3 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.7f);
                g.setComposite(ac3);
                // 对每个字符都用随机颜色
                color = new Color(20 + RandomUtil.num(110), 20 + RandomUtil.num(110), 20 + RandomUtil.num(110));
                g.setColor(color);
                g.drawString(strs[i] + "", (width - (len - i) * w) + size, h - 4);
                color = null;
                ac3 = null;
            }
            ImageIO.write(bi, "png", out);
            out.flush();
            ok = true;
        } catch (IOException e) {
            ok = false;
        } finally {
            IoUtil.close(out);
        }
        return ok;
    }

    /**
     * 画随机码图
     *
     * @param strs 文本
     */
    public BufferedImage graphicsImage(String strs) {
        BufferedImage bi = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        Graphics2D g = (Graphics2D) bi.getGraphics();
        AlphaComposite ac3;
        Color color;
        int len = strs.length();
        g.setColor(Color.WHITE);
        g.fillRect(0, 0, width, height);
        // 随机画干扰圆形
        for (int i = 0; i < 15; i++) {
            color = color(150, 250);
            g.setColor(color);
            // 画椭圆形
            g.drawOval(RandomUtil.num(width), RandomUtil.num(height), 5 + RandomUtil.num(10), 5 + RandomUtil.num(10));
            color = null;
        }
        g.setFont(font);
        int h = height - ((height - font.getSize()) >> 1), w = width / len, size = w - font.getSize() + 1;
        /* 画字符串 */
        for (int i = 0; i < len; i++) {
            // 指定透明度
            ac3 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 0.7f);
            g.setComposite(ac3);
            // 对每个字符都用随机颜色
            color = new Color(20 + RandomUtil.num(110), 20 + RandomUtil.num(110), 20 + RandomUtil.num(110));
            g.setColor(color);
            g.drawString(strs.charAt(i) + "", (width - (len - i) * w) + size, h - 4);
            color = null;
            ac3 = null;
        }
        return bi;
    }

    public void out(OutputStream os, String text) {
        graphicsImage(text.toCharArray(), os);
    }


    /**
     * 生成随机字符数组
     *
     * @return 字符数组
     */
    protected char[] alphas() {
        char[] cs = new char[len];
        for (int i = 0; i < len; i++) {
            cs[i] = RandomUtil.alpha();
        }
        chars = new String(cs);
        return cs;
    }

    public Font getFont() {
        return font;
    }

    public void setFont(Font font) {
        this.font = font;
    }

    public int getLen() {
        return len;
    }

    public void setLen(int len) {
        this.len = len;
    }

    public int getWidth() {
        return width;
    }

    public void setWidth(int width) {
        this.width = width;
    }

    public int getHeight() {
        return height;
    }

    public void setHeight(int height) {
        this.height = height;
    }

    /**
     * 给定范围获得随机颜色
     *
     * @return Color 随机颜色
     */
    protected Color color(int fc, int bc) {
        if (fc > 255) {
            fc = 255;
        }
        if (bc > 255) {
            bc = 255;
        }
        int r = fc + RandomUtil.num(bc - fc);
        int g = fc + RandomUtil.num(bc - fc);
        int b = fc + RandomUtil.num(bc - fc);
        return new Color(r, g, b);
    }

    /**
     * 获取随机字符串
     *
     * @return string
     */
    public String text() {
        return chars;
    }
}
