package cn.morethank.open.admin.common.controller;

import cn.hutool.core.io.IoUtil;
import cn.morethank.open.admin.common.config.CaptchaConfig;
import cn.morethank.open.admin.common.constant.GlobalConstant;
import cn.morethank.open.admin.common.constant.RedisKeyConstant;
import cn.morethank.open.admin.common.domain.ImageCaptcha;
import cn.morethank.open.admin.common.domain.Result;
import cn.morethank.open.admin.common.service.RedisService;
import cn.morethank.open.admin.common.service.RsaService;
import cn.morethank.open.admin.common.util.RandomUtil;
import com.alibaba.fastjson2.JSONObject;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.util.Base64;
import java.util.Map;
import java.util.UUID;

/**
 * 图形验证码的前端控制器
 * @author morethank
 * @since 2022/12/17 17:23
 */
@AllArgsConstructor
@Slf4j
@RestController
public class CaptchaController {

    private final CaptchaConfig captchaConfig;
    private final RedisService redisService;
    private final RsaService rsaService;

    @GetMapping("/captchaImage")
    public Result captcha() {
        ByteArrayOutputStream os = null;
        try {
            // gif格式动画验证码, 宽, 高, 位数。
            int len = captchaConfig.getLen();
            ImageCaptcha captcha = new ImageCaptcha(captchaConfig.getWidth(), captchaConfig.getHeight(), len);
            String code = RandomUtil.alphas(len);
            String uuid = UUID.randomUUID().toString();
            BufferedImage image = captcha.graphicsImage(code);

            os = new ByteArrayOutputStream();
            ImageIO.write(image, GlobalConstant.JPG, os);
            StringBuilder sb = new StringBuilder();
            sb.append(Base64.getEncoder().encodeToString(os.toByteArray()));

            // 过期时间5分钟
            long timeout = GlobalConstant.CACHE_TIMEOUT;
            // 将验证码存入redis
            redisService.hset(RedisKeyConstant.CAPTCHA_KEY, uuid, code, timeout);
            log.info("验证码 -- {} - {}", uuid, code);

            // 生成rsa公钥和私钥
            Map<String, String> rsaKeys = rsaService.getKeys();
            // 私钥放到redis缓存中, 登录时提交
            redisService.hset(GlobalConstant.PRIVATE_KEY, uuid, rsaKeys.get(GlobalConstant.PRIVATE_KEY), timeout);

            JSONObject jsonObject = new JSONObject();
            // 公钥返回前端, 用于前端加密
            jsonObject.put(GlobalConstant.PUBLIC_KEY, rsaKeys.get(GlobalConstant.PUBLIC_KEY));
            jsonObject.put(GlobalConstant.TOKEN, uuid);
            jsonObject.put(GlobalConstant.BASE64_IMG, sb);
            return Result.success(jsonObject);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        } finally {
            IoUtil.close(os);
        }
        return Result.fail();
    }

    /**
     * 生成RSA加密公钥
     * @return 返回公钥
     */
    @GetMapping("/getEncryptKey")
    public Result getEncryptKey() {
        String uuid = UUID.randomUUID().toString();
        // 生成rsa公钥和私钥
        Map<String, String> rsaKeys = rsaService.getKeys();
        // 私钥放到redis缓存中, 过期时间5分钟
        long timeout = GlobalConstant.CACHE_TIMEOUT;
        redisService.hset(GlobalConstant.PRIVATE_KEY, uuid, rsaKeys.get(GlobalConstant.PRIVATE_KEY), timeout);

        JSONObject jsonObject = new JSONObject();
        // 公钥返回前端, 用于前端加密
        jsonObject.put(GlobalConstant.PUBLIC_KEY, rsaKeys.get(GlobalConstant.PUBLIC_KEY));
        jsonObject.put(GlobalConstant.TOKEN, uuid);
        return Result.success(jsonObject);
    }

}
