package cn.montaro.relaxhttp;

import cn.montaro.relaxhttp.annotation.method.ContentType;
import cn.montaro.relaxhttp.annotation.type.BaseUrl;
import cn.montaro.relaxhttp.interfaces.API;
import cn.montaro.relaxhttp.interfaces.adapter.ResultAdapter;
import cn.montaro.relaxhttp.util.ParamUtils;
import okhttp3.*;

import java.io.IOException;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;
import java.util.Map;

/**
 * RelaxHttp 类能通过调用create方法创建API的自定义接口的HTTP实现类
 * 也可直接直接使用静态方法get和post
 * RequestAdapter 用于将RequestRecord 转换成实际请求的Request
 * ResultAdapter 用于将Response 相应转换成方法返回值的类型
 * RequestRecord 保存了继承了API的接口的各类注解数据 请求方式 请求参数名称等
 */
public class RelaxHttp {

    public static final String MEDIA_TYPE_JSON = "application/json; charset=utf-8";
    public static final String MEDIA_TYPE_FORM = "application/x-www-form-urlencoded; charset=utf-8";

    public static ResultAdapter resultAdapter = new DefaultResultAdapter();

    public static OkHttpClient okHttpClient = new OkHttpClient();

    public final static <T extends API> T create(Class<T> api, RequestResolver resolver, OkHttpClient client) {
        if (!api.isInterface())
            throw new IllegalArgumentException("argument must be interface class");
        InvocationHandler handler = RelaxHttp.createProxyObject(api, resolver, client);
        return (T) Proxy.newProxyInstance(api.getClassLoader(), new Class[]{api}, handler);
    }

    public final static <T extends API> T create(Class<T> api, OkHttpClient client) {
        return RelaxHttp.create(api, new DefaultRequestResolver(), client);
    }

    public final static <T extends API> T create(Class<T> api) {
        return RelaxHttp.create(api, new OkHttpClient());
    }

    private static <T extends API> void handleClass(Class<T> api, RequestResolver requestResolver) {
        BaseUrl baseUrl = api.getAnnotation(BaseUrl.class);
        if (baseUrl != null) {
            requestResolver.setBaseUrl(baseUrl.value());
        }
        ContentType contentType = api.getAnnotation(ContentType.class);
        if (contentType != null) {
            requestResolver.setContentType(contentType.value());
        }
    }

    private static <T extends API> RequestResolver createProxyObject(Class<T> api, RequestResolver requestResolver, OkHttpClient client) {
        RelaxHttp.handleClass(api, requestResolver);
        requestResolver.setOkHttpClient(client);
        return requestResolver;
    }

    public static <T> T get(String url, Map<String, Object> params, Class<T> resultType) throws IOException {
        return RelaxHttp.get(url, params, resultType, RelaxHttp.resultAdapter);
    }

    public static <T> T get(String url, Map<String, Object> params, Class<T> resultType, ResultAdapter resultAdapter) throws IOException {
        String realUrl = ParamUtils.addParam(url, params);
        Request request = new Request.Builder().url(realUrl).get().build();
        Response response = RelaxHttp.okHttpClient.newCall(request).execute();
        T result = resultAdapter.onConvert(response, resultType,null);
        return result;
    }

    public static <T> T post(String url, MediaType mediaType, String data, Class<T> resultType) throws IOException {
        return RelaxHttp.post(url, mediaType, data, resultType, RelaxHttp.resultAdapter);
    }

    public static <T> T post(String url, MediaType mediaType, String data, Class<T> resultType, ResultAdapter resultAdapter) throws IOException {
        Request request = new Request.Builder().url(url).post(RequestBody.create(mediaType, data)).build();
        Response response = RelaxHttp.okHttpClient.newCall(request).execute();
        T result = resultAdapter.onConvert(response, resultType,null);
        return result;
    }

}
