package cn.montaro.relaxhttp;

import cn.montaro.relaxhttp.annotation.method.ContentType;
import cn.montaro.relaxhttp.annotation.method.DisableBaseUrl;
import cn.montaro.relaxhttp.annotation.method.request.DELETE;
import cn.montaro.relaxhttp.annotation.method.request.GET;
import cn.montaro.relaxhttp.annotation.method.request.POST;
import cn.montaro.relaxhttp.annotation.method.request.PUT;
import cn.montaro.relaxhttp.annotation.parameter.Body;

import cn.montaro.relaxhttp.annotation.parameter.Header;
import cn.montaro.relaxhttp.annotation.parameter.Path;
import cn.montaro.relaxhttp.annotation.parameter.Query;
import cn.montaro.relaxhttp.annotation.type.BaseUrl;
import cn.montaro.relaxhttp.interfaces.RequestRecord;
import cn.montaro.relaxhttp.util.StringUtils;
import okhttp3.OkHttpClient;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.HashMap;
import java.util.Map;

public class DefaultRequestResolver extends RequestResolver {

    private Map<Method, RequestRecord> cachedRequestRecord = new HashMap<>();

    private Boolean enableCache = true;

    public DefaultRequestResolver() {
        this.setOkHttpClient(new OkHttpClient());
        this.setRequestAdapter(new DefaultRequestAdapter());
        this.setResultAdapter(new DefaultResultAdapter());
    }

    @Override
    public RequestRecord getRequestRecord(Method method, Object[] args) {
        //读取注解数据整合成请求信息 并保存到缓存中
        if (this.enableCache && this.cachedRequestRecord.containsKey(method))
            return this.cachedRequestRecord.get(method);
        DefaultRequestRecord defaultRequestRecord = new DefaultRequestRecord();

        //获取baseUrl
        String baseUrl = this.getMethodBaseUrl(method);
        defaultRequestRecord.baseUrl = baseUrl;

        //获取contentType
        String contentType = this.getContentType(method);
        defaultRequestRecord.contentType = StringUtils.choose(contentType,this.getContentType(),defaultRequestRecord.contentType);

        //获取请求方法和请求路径
        RequestMethodAndPath requestMethodAndPath = this.getRequestMethodAndPath(method);
        defaultRequestRecord.setRequestMethodAndPath(requestMethodAndPath);

        //获取请求参数名称和对应的参数顺序
        this.getRequestParameters(method, defaultRequestRecord);

        if (this.enableCache)
            this.cachedRequestRecord.put(method, defaultRequestRecord);
        return defaultRequestRecord;
    }

    private String getMethodBaseUrl(Method method) {
        DisableBaseUrl disableBaseUrl = method.getAnnotation(DisableBaseUrl.class);
        if (disableBaseUrl != null) return "";
        BaseUrl baseUrl = method.getAnnotation(BaseUrl.class);
        if (baseUrl != null) return baseUrl.value();
        return this.getBaseUrl();
    }

    /**
     * 获取请求方法与路径
     *
     * @param method
     * @return
     */
    private RequestMethodAndPath getRequestMethodAndPath(Method method) {
        GET get = method.getAnnotation(GET.class);
        POST post = method.getAnnotation(POST.class);
        PUT put = method.getAnnotation(PUT.class);
        DELETE delete = method.getAnnotation(DELETE.class);
        if (delete != null)
            return new RequestMethodAndPath(DefaultRequestRecord.METHOD_DELETE, delete.value());
        else if (put != null)
            return new RequestMethodAndPath(DefaultRequestRecord.METHOD_PUT, put.value());
        else if (post != null)
            return new RequestMethodAndPath(DefaultRequestRecord.METHOD_POST, post.value());
        else
            return new RequestMethodAndPath(DefaultRequestRecord.METHOD_GET, get.value());
    }

    private void getRequestParameters(Method method, DefaultRequestRecord requestRecord) {

        Map<String, Integer> pathParameters = new HashMap<>();
        Map<String, Integer> bodyParameters = new HashMap<>();
        Map<String, Integer> queryParameters = new HashMap<>();
        Map<String, Integer> headerParameters = new HashMap<>();
        Parameter[] parameters = method.getParameters();
        for (int i = 0; i < parameters.length; i++) {
            Parameter parameter = parameters[i];
            // java8 可以通过反射获取参数名,但是需使用-parameters参数开启功能
            // 否则获取到的参数名为 arg0 arg1 arg2
            Path path = parameter.getAnnotation(Path.class);
            Body body = parameter.getAnnotation(Body.class);
            Query query = parameter.getAnnotation(Query.class);
            Header header = parameter.getAnnotation(Header.class);
            String paramName = parameter.getName();
            if (path != null) {
                paramName = StringUtils.choose(path.value(), paramName);
                pathParameters.put(paramName, i);
            }
            if (body != null) {
                paramName = StringUtils.choose(body.value(), paramName);
                bodyParameters.put(paramName, i);
            }
            if (query != null) {
                paramName = StringUtils.choose(query.value(), paramName);
                queryParameters.put(paramName, i);
            }
            if (header != null) {
                paramName = StringUtils.choose(header.value(), paramName);
                headerParameters.put(paramName, i);
            }
        }
        requestRecord.bodyParameters = bodyParameters;
        requestRecord.pathParameters = pathParameters;
        requestRecord.queryParameters = queryParameters;
        requestRecord.headerParameters = headerParameters;
    }

    private String getContentType(Method method) {
        ContentType contentType = method.getAnnotation(ContentType.class);
        if (contentType != null)
            return contentType.value();
        return "";
    }

    public Boolean getEnableCache() {
        return enableCache;
    }

    public void setEnableCache(Boolean enableCache) {
        this.enableCache = enableCache;
    }


}
