package cn.minsin.excel.function;

import cn.minsin.core.constant.CharSetConstant;
import cn.minsin.core.exception.MutilsException;
import cn.minsin.core.tools.IOUtil;
import cn.minsin.core.tools.ListUtil;
import cn.minsin.excel.config.MutilsExcelProperties;
import cn.minsin.excel.enums.ExcelVersion;
import cn.minsin.excel.model.create.ExcelCreator;
import cn.minsin.excel.tools.ExcelUtil;
import lombok.Getter;
import lombok.Setter;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletResponse;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * @author: minton.zhang
 * @since: 2020/3/31 23:08
 */
@Getter
@Setter
@Component
public class ExcelCreateFunctions {

    @Autowired
    private MutilsExcelProperties properties;

    private ExcelVersion excelVersion;

    private Workbook workbook;

    private Row row;

    private Sheet sheet;


    public static ExcelCreateFunctions initByVersion(ExcelVersion excelVersion) {
        return new ExcelCreateFunctions().init(excelVersion);
    }

    public static ExcelCreateFunctions initNyInputStream(InputStream in) throws IOException {
        return new ExcelCreateFunctions().init(in);
    }

    public static ExcelCreateFunctions initByTemplatePath(String templatePath, boolean isInDisk) throws IOException {
        return new ExcelCreateFunctions().init(templatePath, isInDisk);
    }


    public ExcelCreateFunctions init(ExcelVersion excelVersion) {
        if (excelVersion == null || excelVersion == ExcelVersion.VERSION_2007) {
            workbook = new XSSFWorkbook();
        } else {
            workbook = new HSSFWorkbook();
        }
        this.excelVersion = excelVersion;
        return this;
    }

    public ExcelCreateFunctions init(InputStream in) throws IOException {
        this.workbook = WorkbookFactory.create(in);
        this.excelVersion = ExcelVersion.checkVersion(workbook);
        return this;
    }

    public ExcelCreateFunctions init(String templatePath, boolean isInDisk) throws IOException {
        InputStream excelTemplate = ExcelUtil.getExcelTemplate(templatePath, isInDisk);
        return init(excelTemplate);
    }

    /**
     * 按照list的规则进行生成excel
     * <p>需要使用{@link cn.minsin.excel.model.create.CellTitle}</p>
     * <p>需要使用{@link cn.minsin.excel.model.create.ExcelCreator}</p>
     *
     * @return
     */
    public ExcelCreateFunctions addRowByList(List<? extends ExcelCreator> listData) throws IllegalAccessException {
        return addRowByList(listData, 0);
    }

    /**
     * 按照list的规则进行生成excel
     * <p>需要使用{@link cn.minsin.excel.model.create.CellTitle}</p>
     * <p>需要使用{@link cn.minsin.excel.model.create.ExcelCreator}</p>
     *
     * @return
     */
    public ExcelCreateFunctions addRowByList(List<? extends ExcelCreator> listData, int startRowIndex) throws IllegalAccessException {
        MutilsException.throwException(ListUtil.isEmpty(listData), "需要生成Excel的数据不能为空");
        //创建excel
        for (int i = 0; i < listData.size(); i++) {
            ExcelCreator excelCreator = listData.get(i);
            excelCreator.create(this, i + startRowIndex);
        }
        return this;
    }


    public ExcelCreateFunctions sheet(int sheetnum) {
        try {
            sheet = workbook.getSheetAt(sheetnum);
        } catch (Exception e) {
            sheet = workbook.createSheet();
            workbook.setActiveSheet(sheetnum);
        }
        return this;
    }

    public ExcelCreateFunctions sheet(int sheetNum, String name) {
        try {
            sheet = workbook.getSheetAt(sheetNum);
        } catch (Exception e) {
            sheet = workbook.createSheet(name);
            workbook.setActiveSheet(sheetNum);
        }
        return this;
    }

    public ExcelCreateFunctions row(int rowNum) {
        // 初始化row
        row = null;
        row = sheet.getRow(rowNum);
        if (row == null) {
            row = sheet.createRow(rowNum);
        }
        return this;
    }

    public ExcelCreateFunctions cell(int index, Object value) {
        Cell cell;
        try {
            cell = row.getCell(index);
            cell.setCellType(CellType.STRING);
        } catch (Exception e) {
            cell = row.createCell(index, CellType.STRING);
        }
        if (value == null) {
            cell.setCellValue("");
            return this;
        } else if (value instanceof String) {
            cell.setCellValue((String) value);
        } else if (value instanceof Integer) {
            cell.setCellValue((Integer) value);
        } else if (value instanceof Boolean) {
            cell.setCellValue((Boolean) value);
        } else if (value instanceof Calendar) {
            cell.setCellValue((Calendar) value);
        } else if (value instanceof Date) {
            cell.setCellValue((Date) value);
        } else {
            cell.setCellValue(value.toString());
        }

        return this;
    }

    /**
     * D://upload/aaa
     *
     * @param filename 无后缀的文件名 2018年10月11日
     * @throws IOException
     */
    public void export(String filename) throws IOException {
        FileOutputStream fileOutputStream = null;
        try {
            fileOutputStream = new FileOutputStream(filename + excelVersion.getSuffix());
            workbook.write(fileOutputStream);
        } finally {
            IOUtil.close(workbook, fileOutputStream);
        }
    }

    /**
     * 导出文件到浏览器
     *
     * @param resp
     * @param fileName 无后缀的文件名 2018年10月11日
     */
    public void export(HttpServletResponse resp, String fileName, boolean isUTF8) {
        try {
            if (isUTF8) {
                fileName = new String(fileName.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1) + excelVersion.getSuffix();
                resp.setCharacterEncoding(CharSetConstant.UTF_8);
            }
            resp.setContentType("application/x-msdownload; charset=utf-8");
            resp.setHeader("content-disposition", "attachment;filename=" + fileName);
            workbook.write(resp.getOutputStream());
        } catch (Exception e) {
            error(resp, "excel导出失败，已切换为错误模板", e);
        } finally {
            IOUtil.close(workbook);
        }
    }


    public void error(HttpServletResponse resp, String message, Exception error) {
        //通过new的方式进行创建时需要使用
        if (properties == null) {
            properties = new MutilsExcelProperties();
        }
        try {
            String errorMessage = error == null ? "" : error.getMessage();
            this.sheet(properties.getErrorTemplateSheetIndex())
                    .row(properties.getErrorTemplateRowIndex())
                    .cell(properties.getErrorTemplateCellIndex(), message + "\n\n" + errorMessage)
                    .export(resp, properties.getErrorTemplateExportName(), true);
        } catch (Exception e) {
            init(ExcelVersion.VERSION_2007).export(resp, properties.getErrorTemplateExportName(), true);
        }
    }

}
