package cn.mapway.ui.server.apt;


import com.google.auto.service.AutoService;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import javax.tools.JavaFileObject;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Set;
import java.util.logging.Logger;

/**
 * JsStoreAnnotationProcessor
 * 处理 @JsStoreAnnotation
 *
 * @author zhang
 */
@SupportedAnnotationTypes(
        "cn.mapway.ui.server.apt.jsstore.JsStoreTable")
@SupportedSourceVersion(SourceVersion.RELEASE_8)
@AutoService(javax.annotation.processing.Processor.class)
public class JsStoreAnnotationProcessor extends AbstractProcessor {
    private static final Logger logger = Logger.getLogger(JsStoreAnnotationProcessor.class.getName());

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        StringBuilder sb = new StringBuilder();
        // Check if it's the last round of processing
        // This prevents the processor from running indefinitely if no annotations are found in subsequent rounds
        if (roundEnv.processingOver()) {
            return false;
        }

        // It's good practice to log or print something when the processor is actually invoked
        // rather than only when annotations are found.
        System.out.println("JsStoreAnnotationProcessor: Processing round...");


        for (TypeElement t : annotations) {
            sb.append("\t").append(t.getQualifiedName());
            Set<? extends Element> elements = roundEnv.getElementsAnnotatedWith(t);
            if (elements != null) {
                for (Element e : elements) {
                    sb.append("\n\t").append(e.toString());
                    handleClass(e);
                }
            }
        }

        System.out.println("JsStoreAnnotationProcessor: Finished processing. Annotations found: " + sb);
        return true; // Return true to claim the annotations and prevent other processors from handling them
    }

    private void handleClass(Element e) {
        if (e.getKind() != ElementKind.CLASS) {
            // Log a warning or error if an annotation is found on a non-class element,
            // or ensure @Target on JsStoreTable prevents this.
            return;
        }
        TypeElement typeElement = (TypeElement) e;
        String className = typeElement.getQualifiedName().toString();
        String packageName = null;
        int lastDot = className.lastIndexOf('.');
        if (lastDot > 0) {
            packageName = className.substring(0, lastDot);
        }

        String simpleClassName = className.substring(lastDot + 1);
        String builderClassName = className + "Builder";
        String builderSimpleClassName = builderClassName
                .substring(lastDot + 1);

        JavaFileObject builderFile = null;
        try {
            // Get a Filer to create new source files
            builderFile = processingEnv.getFiler()
                    .createSourceFile(builderClassName, typeElement); // Pass element for proper dependency tracking
        } catch (IOException ex) {
            // Log the error instead of just printing stack trace
            logger.severe("Error creating source file " + builderClassName + ": " + ex.getMessage());
            return;
        }

        try (PrintWriter out = new PrintWriter(builderFile.openWriter())) {
            if (packageName != null) {
                out.print("package ");
                out.print(packageName);
                out.println(";");
                out.println();
            }

            // You're generating an empty class. This is usually the starting point for more complex code generation.
            out.print("public class ");
            out.print(builderSimpleClassName);
            out.println(" {");
            out.println("    // Generated by JsStoreAnnotationProcessor");
            out.println("}");
            logger.info("Generated class: " + builderClassName); // Log successful generation
        } catch (IOException ex) {
            logger.severe("Error writing to source file " + builderClassName + ": " + ex.getMessage());
        }
    }
}