package cn.majingjing.core.util;


import java.io.File;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Objects;
import java.util.jar.JarEntry;

/**
 * Package scan utils
 *
 * @author JingjingMa
 * @date 2019/8/7 16:25
 */
public class PackageUtils {

    public static List<Class<?>> scan(String packageName) throws IOException, URISyntaxException, ClassNotFoundException {
        // convert package name to path
        String packagePath = packageName.replace(".", "/");

        // get the enumeration of urls by the class loader
        Enumeration<URL> resources = Thread.currentThread()
                .getContextClassLoader()
                .getResources(packagePath);

        List<Class<?>> classList = new ArrayList<>();
        while (resources.hasMoreElements()) {
            URL url = resources.nextElement();

            // handle jar packages
            if ("jar".equals(url.getProtocol())) {
                List<Class<?>> jarClassList = parseJarFile(url, packageName);
                classList.addAll(jarClassList);
            } else {
                File file = new File(url.toURI());

                if (file.exists()) {
                    // handle simple packages
                    List<Class<?>> simpleClassList = parseClassFile(file, packageName);
                    classList.addAll(simpleClassList);
                }
            }
        }
        return classList;
    }

    /**
     * parse jar package
     *
     * @param url         the jar url
     * @param packageName package name
     * @throws IOException if parse failed will be throw IOException
     */
    private static List<Class<?>> parseJarFile(URL url, String packageName) throws IOException, ClassNotFoundException {
        List<Class<?>> classList = new ArrayList<>();
        Enumeration<JarEntry> jarEntries = ((JarURLConnection) url.openConnection())
                .getJarFile().entries();

        while (jarEntries.hasMoreElements()) {
            JarEntry jarEntry = jarEntries.nextElement();
            String jarName = jarEntry.getName();

            if (!jarEntry.isDirectory() && jarName.endsWith(".class")) {
                // convert file path name to package name
                String className = jarName.replace("/", ".").replace(".class", "");
                if (className.startsWith(packageName)) {
                    Class<?> clazz = dealClassName(className);
                    classList.add(clazz);
                }

            }
        }
        return classList;
    }

    /**
     * parse simple package
     *
     * @param curFile     file
     * @param packageName package name
     */
    private static List<Class<?>> parseClassFile(File curFile, String packageName) throws IOException, ClassNotFoundException {
        List<Class<?>> classList = new ArrayList<>();
        // Filter folders and class files
        File[] fileList = curFile.listFiles(pathname -> pathname.isDirectory() || pathname.getName().endsWith(".class"));

        for (File file : Objects.requireNonNull(fileList)) {
            String fileName = file.getName();
            if (file.isDirectory()) {
                List<Class<?>> classFileList = parseClassFile(file, packageName + "." + fileName);
                classList.addAll(classFileList);
            } else {
                String className = packageName + "." + fileName.replace(".class", "");
                Class<?> clz = dealClassName(className);
                classList.add(clz);
            }
        }
        return classList;
    }

    /**
     * handler the find class
     *
     * @param className class name
     */
    private static Class<?> dealClassName(String className) throws IOException, ClassNotFoundException {
        return Class.forName(className);
    }

}
