package cn.majingjing.core.util;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * Common tool methods
 *
 * @author majingjing
 */
public class MyUtils {
    /**
     * Brace placeholder
     */
    private static final String BRACKETS_PLACE_HOLDER = "{}";

    /**
     * If obj is Null, return the default given value
     *
     * @param obj        object
     * @param defaultObj given default value
     * @param <T>        the object type
     * @return obj or defaultObj
     */
    public static <T> T objectNull2Default(T obj, T defaultObj) {
        return MyUtils.isNull(obj) ? defaultObj : obj;
    }

    /**
     * Compare the two object is equal
     *
     * @param a a
     * @param b b
     * @return bool
     */
    public static boolean equals(Object a, Object b) {
        if (a instanceof Number && b instanceof Number) {
            return MyUtils.toStrings(a).equals(MyUtils.toStrings(b));
        }
        return Objects.equals(a, b);
    }

    /**
     * Check the object is null
     *
     * @param obj object
     * @return bool (is null will return true , or return false)
     */
    public static boolean isNull(Object obj) {
        return (null == obj);
    }

    /**
     * Check the object is not null
     *
     * @param obj object
     * @return bool (is not null will return true , or return false)
     * @see MyUtils#isNull(Object)
     */
    public static boolean isNotNull(Object obj) {
        return !isNull(obj);
    }

    /**
     * Check the given obj is null or empty
     * <br/>
     *
     * <ul>
     * <li>
     * null ---> true
     * </li>
     * <li>
     * ""  or "&nbsp;"---> true
     * </li>
     * <li>
     * (Integer)0 ---> true
     * </li>
     * <li>
     * (List.size()==0)list ---> true
     * </li>
     * <li>
     * ([])array ---> true
     * </li>
     * </ul>
     *
     * @param obj object
     * @return null or empty will be return true, or return false
     */
    public static boolean isEmpty(Object obj) {
        if (obj == null) {
            return true;
        }
        if (obj instanceof String) {
            return ("".equals(((String) obj).trim()));
        }
        if (obj instanceof Number) {
            return (((Number) obj).intValue() == 0);
        }
        if (obj instanceof Collection) {
            return ((Collection) obj).isEmpty();
        }
        if (obj.getClass().isArray()) {
            return ((Object[]) obj).length == 0;
        }
        throw new UnsupportedOperationException();
    }

    /**
     * Check the given obj is not ( null or empty)
     *
     * @param obj object
     * @return bool
     * @see MyUtils#isEmpty(Object)
     */
    public static boolean isNotEmpty(Object obj) {
        return !isEmpty(obj);
    }

    /**
     * Convert Object to Long type
     *
     * @param obj object
     * @return object of type long
     */
    public static Long toLong(Object obj) {
        if (obj instanceof Number) {
            return ((Number) obj).longValue();
        }
        return Long.valueOf(String.valueOf(obj));
    }

    /**
     * Convert Object to Integer type
     *
     * @param obj object
     * @return Integer
     */
    public static Integer toInteger(Object obj) {
        if (obj instanceof Number) {
            return ((Number) obj).intValue();
        }
        return Integer.valueOf(String.valueOf(obj));
    }

    /**
     * Convert s to a new string based on obj(s)
     * <pre>
     * toStrings("Welcome: {}, Now is {}", "Marion",new Date());
     * result: Welcome: Marion, Now is Tue Mar 21 21:55:44 CST 2017
     * </pre>
     * @param s the convert str
     * @param obj value to be replaced
     * @return a new string
     */
    public static String toStrings(String s, Object... obj) {
        StringBuilder sb = new StringBuilder(s);
        for (int i = 0, m = sb.indexOf(BRACKETS_PLACE_HOLDER); i < obj.length && m > -1; i++, m = sb.indexOf(BRACKETS_PLACE_HOLDER)) {
            sb.delete(m, m + 2).insert(m, obj[i]);
        }
        return sb.toString();
    }

    /**
     * Convert Object to String ,null will be convert to ""
     *
     * @param obj Object
     * @return String
     */
    public static String toStrings(Object obj) {
        if (null == obj) {
            return "";
        }
        if (obj instanceof String) {
            return (String) obj;
        }
        if (obj instanceof byte[]) {
            return new String((byte[]) obj);
        }
        return String.valueOf(obj);

    }


    /**
     * object to map
     *
     * @param obj object will be converted
     * @return map map object
     * @throws IntrospectionException    IntrospectionException
     * @throws IllegalArgumentException  IllegalArgumentException
     * @throws IllegalAccessException    IllegalAccessException
     * @throws InvocationTargetException InvocationTargetException
     */
    public static Map<String, Object> toMap(Object obj)
            throws IntrospectionException, IllegalArgumentException, IllegalAccessException, InvocationTargetException {
        Map<String, Object> map = new HashMap<>(16);
        BeanInfo beanInfo = Introspector.getBeanInfo(obj.getClass());
        PropertyDescriptor[] propertyDescriptors = beanInfo.getPropertyDescriptors();
        for (PropertyDescriptor property : propertyDescriptors) {
            if (!"class".equals(property.getName())) {
                Object realVal = property.getReadMethod().invoke(obj);
                if (realVal != null) {
                    map.put(property.getName(), realVal);
                } else {
                    map.put(property.getName(), "");
                }
            }
        }
        return map;
    }


    /**
     * Check the current system is window
     *
     * @return bool
     */
    public static boolean isWindowsOS() {
        return System.getProperty("os.name").toLowerCase().indexOf("window") >= 0;
    }

    /**
     * Check the current system is linux
     *
     * @return bool
     */
    public static boolean isLinuxOS() {
        return System.getProperty("os.name").toLowerCase().indexOf("linux") >= 0;
    }

    /**
     * Check is image type name
     *
     * @param s string
     * @return bool
     */
    public static boolean isImageName(String s) {
        return "bmp,jpg,png,gif,image/png,image/jpeg".contains(s.toLowerCase());
    }

    /**
     * To byte array
     *
     * @param obj object
     * @return byte array
     */
    public static byte[] toBytes(String obj) {
        if (null == obj) {
            return null;
        }
        try {
            return obj.getBytes("utf-8");
        } catch (Exception e) {
            return null;
        }
    }


}
