package cn.majingjing.core.util;


import java.security.SecureRandom;
import java.util.Random;
import java.util.UUID;

/**
 * @author apple
 */
public final class IdUtils {
    private IdUtils(){}

    /**
     * Get a UUID string(LowerCase), length=36
     *
     * @return String
     */
    public static String uuid() {
        return UUID.randomUUID().toString().toLowerCase();
    }

    /**
     * Get a UUID string(UpperCase), length=36
     *
     * @return String
     */
    public static String UUID() {
        return UUID.randomUUID().toString().toUpperCase();
    }

    /**
     * Get a GUID string(UpperCase), length=32
     *
     * @return String
     */
    public static String GUID() {
        return UUID().replace("-", "");
    }

    /**
     * Get a UUID string(LowerCase), length=32
     *
     * @return String
     */
    public static String guid() {
        return uuid().replace("-", "");
    }


    /**
     * 雪花算法id
     * @return 数字,length=18
     */
    public static long snowflakeId(){
        return Instance.SnowflakeId.nextId();
    }

    /**
     * 随机字符串
     * @return 字符串,length=21
     */
    public static String nanoId(){
        return NanoId.randomNanoId();
    }

    /**
     * 随机字符串(指定长度)
     * @param length 指定长度
     * @return 字符串
     */
    public static String nanoId(int length){
        return NanoId.randomNanoId(length);
    }

    private static class Instance {
        static SnowflakeId SnowflakeId = new SnowflakeId(1L, 1L);
    }

    private static final class NanoId {

        /**
         * <code>NanoId</code> instances should NOT be constructed in standard programming.
         * Instead, the class should be used as <code>NanoIdUtils.randomNanoId();</code>.
         */
        private NanoId() {
            //Do Nothing
        }

        /**
         * The default random number generator used by this class.
         * Creates cryptographically strong NanoId Strings.
         */
        public static final SecureRandom DEFAULT_NUMBER_GENERATOR = new SecureRandom();

        /**
         * The default alphabet used by this class.
         * Creates url-friendly NanoId Strings using 64 unique symbols.
         */
        public static final char[] DEFAULT_ALPHABET =
//                    "_-0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
//                "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
                "0123456789abcdefghijklmnopqrstuvwxyz"
                        .toCharArray();

        /**
         * The default size used by this class.
         * Creates NanoId Strings with slightly more unique values than UUID v4.
         */
        public static final int DEFAULT_SIZE = 21;

        /**
         * Static factory to retrieve a url-friendly, pseudo randomly generated, NanoId String.
         *
         * The generated NanoId String will have 21 symbols.
         *
         * The NanoId String is generated using a cryptographically strong pseudo random number
         * generator.
         *
         * @return A randomly generated NanoId String.
         */
        public static String randomNanoId() {
            return randomNanoId(DEFAULT_NUMBER_GENERATOR, DEFAULT_ALPHABET, DEFAULT_SIZE);
        }

        public static String randomNanoId(int length) {
            return randomNanoId(DEFAULT_NUMBER_GENERATOR, DEFAULT_ALPHABET, length);
        }

        /**
         * Static factory to retrieve a NanoId String.
         *
         * The string is generated using the given random number generator.
         *
         * @param random   The random number generator.
         * @param alphabet The symbols used in the NanoId String.
         * @param size     The number of symbols in the NanoId String.
         * @return A randomly generated NanoId String.
         */
        public static String randomNanoId(final Random random, final char[] alphabet, final int size) {

            if (random == null) {
                throw new IllegalArgumentException("random cannot be null.");
            }

            if (alphabet == null) {
                throw new IllegalArgumentException("alphabet cannot be null.");
            }

            if (alphabet.length == 0 || alphabet.length >= 256) {
                throw new IllegalArgumentException("alphabet must contain between 1 and 255 symbols.");
            }

            if (size <= 0) {
                throw new IllegalArgumentException("size must be greater than zero.");
            }

            final int mask = (2 << (int) Math.floor(Math.log(alphabet.length - 1) / Math.log(2))) - 1;
            final int step = (int) Math.ceil(1.6 * mask * size / alphabet.length);

            final StringBuilder idBuilder = new StringBuilder();

            while (true) {

                final byte[] bytes = new byte[step];
                random.nextBytes(bytes);

                for (int i = 0; i < step; i++) {

                    final int alphabetIndex = bytes[i] & mask;

                    if (alphabetIndex < alphabet.length) {
                        idBuilder.append(alphabet[alphabetIndex]);
                        if (idBuilder.length() == size) {
                            return idBuilder.toString();
                        }
                    }

                }

            }

        }
    }


    /**
     *
     * Twitter 开源的 Snowflake 算法
     *
     * 参考自 https://gavinlee1.github.io/2017/06/28/%E5%B8%B8%E8%A7%81%E5%88%86%E5%B8%83%E5%BC%8F%E5%85%A8%E5%B1%80%E5%94%AF%E4%B8%80ID%E7%94%9F%E6%88%90%E7%AD%96%E7%95%A5%E5%8F%8A%E7%AE%97%E6%B3%95%E7%9A%84%E5%AF%B9%E6%AF%94/
     *
     */
    private static final class SnowflakeId {

        //================================================Algorithm's Parameter=============================================
        // 系统开始时间截 (UTC 2017-06-28 00:00:00)
        private final long startTime = 1498608000000L;
        // 机器id所占的位数
        private final long workerIdBits = 5L;
        // 数据标识id所占的位数
        private final long dataCenterIdBits = 5L;
        // 支持的最大机器id(十进制)，结果是31 (这个移位算法可以很快的计算出几位二进制数所能表示的最大十进制数)
        // -1L 左移 5位 (worker id 所占位数) 即 5位二进制所能获得的最大十进制数 - 31
        private final long maxWorkerId = -1L ^ (-1L << workerIdBits);
        // 支持的最大数据标识id - 31
        private final long maxDataCenterId = -1L ^ (-1L << dataCenterIdBits);
        // 序列在id中占的位数
        private final long sequenceBits = 12L;
        // 机器ID 左移位数 - 12 (即末 sequence 所占用的位数)
        private final long workerIdMoveBits = sequenceBits;
        // 数据标识id 左移位数 - 17(12+5)
        private final long dataCenterIdMoveBits = sequenceBits + workerIdBits;
        // 时间截向 左移位数 - 22(5+5+12)
        private final long timestampMoveBits = sequenceBits + workerIdBits + dataCenterIdBits;
        // 生成序列的掩码(12位所对应的最大整数值)，这里为4095 (0b111111111111=0xfff=4095)
        private final long sequenceMask = -1L ^ (-1L << sequenceBits);
        //=================================================Works's Parameter================================================
        /**
         * 工作机器ID(0~31)
         */
        private long workerId;
        /**
         * 数据中心ID(0~31)
         */
        private long dataCenterId;
        /**
         * 毫秒内序列(0~4095)
         */
        private long sequence = 0L;
        /**
         * 上次生成ID的时间截
         */
        private long lastTimestamp = -1L;
        //===============================================Constructors=======================================================
        /**
         * 构造函数
         *
         * @param workerId     工作ID (0~31)
         * @param dataCenterId 数据中心ID (0~31)
         */
        public SnowflakeId(long workerId, long dataCenterId) {
            if (workerId > maxWorkerId || workerId < 0) {
                throw new IllegalArgumentException(String.format("Worker Id can't be greater than %d or less than 0", maxWorkerId));
            }
            if (dataCenterId > maxDataCenterId || dataCenterId < 0) {
                throw new IllegalArgumentException(String.format("DataCenter Id can't be greater than %d or less than 0", maxDataCenterId));
            }
            this.workerId = workerId;
            this.dataCenterId = dataCenterId;
        }
        // ==================================================Methods========================================================
        /**
         * 线程安全的获得下一个 ID 的方法
         * @return
         */
        public synchronized long nextId() {
            long timestamp = currentTime();
            //如果当前时间小于上一次ID生成的时间戳: 说明系统时钟回退过 - 这个时候应当抛出异常
            if (timestamp < lastTimestamp) {
                throw new RuntimeException(
                        String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds", lastTimestamp - timestamp));
            }
            //如果是同一时间生成的，则进行毫秒内序列
            if (lastTimestamp == timestamp) {
                sequence = (sequence + 1) & sequenceMask;
                //毫秒内序列溢出 即 序列 > 4095
                if (sequence == 0) {
                    //阻塞到下一个毫秒,获得新的时间戳
                    timestamp = blockTillNextMillis(lastTimestamp);
                }
            }
            //时间戳改变，毫秒内序列重置
            else {
                sequence = 0L;
            }
            //上次生成ID的时间截
            lastTimestamp = timestamp;
            //移位并通过或运算拼到一起组成64位的ID
            return ((timestamp - startTime) << timestampMoveBits)
                    | (dataCenterId << dataCenterIdMoveBits)
                    | (workerId << workerIdMoveBits)
                    | sequence;
        }
        /**
         * 阻塞到下一个毫秒 即 直到获得新的时间戳
         * @param lastTimestamp
         * @return
         */
        protected long blockTillNextMillis(long lastTimestamp) {
            long timestamp = currentTime();
            while (timestamp <= lastTimestamp) {
                timestamp = currentTime();
            }
            return timestamp;
        }

        /**
         * 获得以毫秒为单位的当前时间
         * @return
         */
        protected long currentTime() {
            return System.currentTimeMillis();
        }

    }

}
