/*
 * Decompiled with CFR 0.152.
 */
package com.relaxed.pool.monitor.monitor;

import cn.hutool.core.thread.ThreadUtil;
import com.relaxed.pool.monitor.AlertService;
import com.relaxed.pool.monitor.ThreadPoolMonitorProperties;
import com.relaxed.pool.monitor.monitor.MonitoredThreadPool;
import com.relaxed.pool.monitor.monitor.ThreadPoolStats;
import com.relaxed.pool.monitor.monitor.ThreadPoolTrend;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ThreadPoolTaskMonitor {
    private static final Logger log = LoggerFactory.getLogger(ThreadPoolTaskMonitor.class);
    private final Map<String, MonitoredThreadPool> threadPoolMap = new ConcurrentHashMap<String, MonitoredThreadPool>();
    private final Map<String, List<ThreadPoolStats>> historyStats = new ConcurrentHashMap<String, List<ThreadPoolStats>>();
    private final ThreadPoolMonitorProperties monitorProperties;
    private final AlertService alertService;

    public ThreadPoolTaskMonitor(AlertService alertService, ThreadPoolMonitorProperties monitorProperties) {
        this.alertService = alertService;
        this.monitorProperties = monitorProperties;
        if (this.monitorProperties.isMonitorEnabled()) {
            this.startMonitorThread();
        }
    }

    public <T extends ThreadPoolExecutor> ThreadPoolExecutor register(String name, T executor) {
        if (!this.monitorProperties.isMonitorEnabled()) {
            return executor;
        }
        MonitoredThreadPool monitoredPool = new MonitoredThreadPool(name, executor);
        return this.register(name, monitoredPool);
    }

    public ThreadPoolExecutor register(String name, MonitoredThreadPool monitoredPool) {
        ThreadPoolExecutor executor = monitoredPool.getOriginalExecutor();
        this.threadPoolMap.put(name, monitoredPool);
        this.historyStats.put(name, new ArrayList());
        log.info("\u7ebf\u7a0b\u6c60 [{}] \u5df2\u6ce8\u518c\u5230\u76d1\u63a7\u7cfb\u7edf, \u6838\u5fc3\u7ebf\u7a0b\u6570: {}, \u6700\u5927\u7ebf\u7a0b\u6570: {}, \u961f\u5217\u7c7b\u578b: {}, \u961f\u5217\u5bb9\u91cf: {}", new Object[]{name, executor.getCorePoolSize(), executor.getMaximumPoolSize(), executor.getQueue().getClass().getSimpleName(), this.getQueueCapacity(executor.getQueue())});
        return monitoredPool;
    }

    private void startMonitorThread() {
        Thread thread = new Thread(new Runnable(){

            @Override
            public void run() {
                while (true) {
                    try {
                        ThreadPoolTaskMonitor.this.collectStats();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    ThreadUtil.sleep((long)ThreadPoolTaskMonitor.this.monitorProperties.getMonitorIntervalMills());
                }
            }
        });
        thread.setName("pool-monitor-daemon");
        thread.setDaemon(true);
        thread.start();
    }

    public void collectStats() {
        if (!this.monitorProperties.isMonitorEnabled() || this.threadPoolMap.isEmpty()) {
            return;
        }
        this.threadPoolMap.forEach((name, pool) -> {
            ThreadPoolStats stats = new ThreadPoolStats();
            stats.setTimestamp(System.currentTimeMillis());
            stats.setPoolName((String)name);
            stats.setActiveThreads(pool.getActiveCount());
            stats.setCorePoolSize(pool.getCorePoolSize());
            stats.setMaximumPoolSize(pool.getMaximumPoolSize());
            stats.setLargestPoolSize(pool.getLargestPoolSize());
            stats.setPoolSize(pool.getPoolSize());
            stats.setQueueSize(pool.getQueue().size());
            stats.setQueueCapacity(this.getQueueCapacity(pool.getQueue()));
            stats.setTaskCount(pool.getTaskCount());
            stats.setCompletedTaskCount(pool.getCompletedTaskCount());
            stats.setRejectedCount(pool.getRejectedCount());
            this.calculateMetrics(stats);
            List<ThreadPoolStats> history = this.historyStats.get(name);
            history.add(stats);
            if (history.size() > 50) {
                history.remove(0);
            }
            this.checkAlert(stats);
            log.debug("\u7ebf\u7a0b\u6c60 [{}] \u72b6\u6001: \u6d3b\u8dc3\u7ebf\u7a0b {}/{} ({}%), \u961f\u5217 {}/{} ({}%), \u5df2\u5b8c\u6210\u4efb\u52a1 {}, \u62d2\u7edd\u4efb\u52a1 {}", new Object[]{name, stats.getActiveThreads(), stats.getMaximumPoolSize(), stats.getActiveThreadRatio(), stats.getQueueSize(), stats.getQueueCapacity(), stats.getQueueUsageRatio(), stats.getCompletedTaskCount(), stats.getRejectedCount()});
            if (stats.getActiveThreadRatio() > 90.0 && stats.getQueueUsageRatio() > 70.0) {
                this.autoAdjustThreadPool((MonitoredThreadPool)pool);
            }
            if (this.isRestoreThreadPool((MonitoredThreadPool)pool, stats)) {
                this.restoreThreadPool((MonitoredThreadPool)pool);
            }
        });
    }

    private boolean isRestoreThreadPool(MonitoredThreadPool pool, ThreadPoolStats stats) {
        long idleRatioMaxThreshold = this.monitorProperties.getIdleRatioMaxThreshold();
        long idleRatioIntervalMills = this.monitorProperties.getIdleRatioIntervalMills();
        int originalMaximumPoolSize = pool.getOriginalExecutor().getMaximumPoolSize();
        int maximumPoolSize = pool.getMaximumPoolSize();
        return maximumPoolSize > originalMaximumPoolSize && stats.getActiveThreadRatio() < (double)idleRatioMaxThreshold && stats.getQueueUsageRatio() < (double)idleRatioMaxThreshold && System.currentTimeMillis() - pool.getLastUpdateTimeMills() > idleRatioIntervalMills;
    }

    private void restoreThreadPool(MonitoredThreadPool executor) {
        String poolName = this.getPoolName(executor);
        int originalMaximumPoolSize = executor.getOriginalExecutor().getMaximumPoolSize();
        int maximumPoolSize = executor.getMaximumPoolSize();
        log.info("\u81ea\u52a8\u8c03\u6574\u7ebf\u7a0b\u6c60 [{}] \u6062\u590d\u539f\u7ebf\u7a0b\u6570: {} -> {}", new Object[]{poolName, maximumPoolSize, originalMaximumPoolSize});
        executor.setMaximumPoolSize(originalMaximumPoolSize);
        executor.setLastUpdateTimeMills(System.currentTimeMillis());
    }

    public ThreadPoolTrend getTrend(String poolName) {
        List history = this.historyStats.getOrDefault(poolName, Collections.emptyList());
        if (history.isEmpty()) {
            return null;
        }
        ThreadPoolTrend trend = new ThreadPoolTrend();
        trend.setPoolName(poolName);
        trend.setAvgMaximumPoolSize(history.stream().mapToDouble(ThreadPoolStats::getMaximumPoolSize).average().orElse(0.0));
        trend.setAvgActiveThreadRatio(history.stream().mapToDouble(ThreadPoolStats::getActiveThreadRatio).average().orElse(0.0));
        trend.setAvgQueueUsageRatio(history.stream().mapToDouble(ThreadPoolStats::getQueueUsageRatio).average().orElse(0.0));
        trend.setTotalRejectedCount(history.stream().mapToLong(ThreadPoolStats::getRejectedCount).sum());
        trend.setTaskCompletionRatio(history.stream().mapToDouble(ThreadPoolStats::getTaskCompletionRatio).average().orElse(100.0));
        return trend;
    }

    public List<ThreadPoolStats> getAllPoolStats() {
        return this.threadPoolMap.keySet().stream().map(name -> {
            List history = this.historyStats.getOrDefault(name, Collections.emptyList());
            return history.isEmpty() ? null : (ThreadPoolStats)history.get(history.size() - 1);
        }).filter(Objects::nonNull).collect(Collectors.toList());
    }

    private void calculateMetrics(ThreadPoolStats stats) {
        stats.setActiveThreadRatio(stats.getMaximumPoolSize() > 0 ? (double)stats.getActiveThreads() / (double)stats.getMaximumPoolSize() * 100.0 : 0.0);
        stats.setQueueUsageRatio(stats.getQueueCapacity() > 0 ? (double)stats.getQueueSize() / (double)stats.getQueueCapacity() * 100.0 : 0.0);
        stats.setTaskCompletionRatio(stats.getTaskCount() > 0L ? (double)stats.getCompletedTaskCount() / (double)stats.getTaskCount() * 100.0 : 100.0);
    }

    private void checkAlert(ThreadPoolStats stats) {
        boolean needAlert = false;
        StringBuilder alertMsg = new StringBuilder();
        Integer alertThreshold = this.monitorProperties.getAlertThreshold();
        if (stats.getActiveThreadRatio() > (double)alertThreshold.intValue()) {
            needAlert = true;
            alertMsg.append("\u6d3b\u8dc3\u7ebf\u7a0b\u6bd4\u4f8b\u8fc7\u9ad8: ").append(String.format("%.1f%%", stats.getActiveThreadRatio())).append("; ");
        }
        if (stats.getQueueUsageRatio() > (double)alertThreshold.intValue()) {
            needAlert = true;
            alertMsg.append("\u961f\u5217\u4f7f\u7528\u7387\u8fc7\u9ad8: ").append(String.format("%.1f%%", stats.getQueueUsageRatio())).append("; ");
        }
        if (stats.getRejectedCount() > 0L) {
            needAlert = true;
            alertMsg.append("\u5b58\u5728\u4efb\u52a1\u88ab\u62d2\u7edd: ").append(stats.getRejectedCount()).append("\u4e2a; ");
        }
        if (needAlert) {
            String finalMsg = String.format("\u7ebf\u7a0b\u6c60\u544a\u8b66 [%s]: %s", stats.getPoolName(), alertMsg);
            this.alertService.sendAlert(finalMsg, this.monitorProperties.getAlertChannels().split(","));
        }
    }

    private void autoAdjustThreadPool(MonitoredThreadPool executor) {
        if (!this.monitorProperties.isAdjustPoolNumEnabled()) {
            return;
        }
        int adjustPoolMaxinumThreshold = this.monitorProperties.getAdjustPoolMaxinumThreshold();
        int currentMax = executor.getMaximumPoolSize();
        if (currentMax >= adjustPoolMaxinumThreshold) {
            return;
        }
        int newMax = Math.min(currentMax + 5, currentMax * 2);
        if (newMax > adjustPoolMaxinumThreshold) {
            newMax = adjustPoolMaxinumThreshold;
        }
        log.info("\u81ea\u52a8\u8c03\u6574\u7ebf\u7a0b\u6c60 [{}] \u6700\u5927\u7ebf\u7a0b\u6570: {} -> {}", new Object[]{this.getPoolName(executor), currentMax, newMax});
        executor.setMaximumPoolSize(newMax);
        executor.setLastUpdateTimeMills(System.currentTimeMillis());
    }

    private String getPoolName(ThreadPoolExecutor executor) {
        for (Map.Entry<String, MonitoredThreadPool> entry : this.threadPoolMap.entrySet()) {
            if (entry.getValue() != executor) continue;
            return entry.getKey();
        }
        return "unknown";
    }

    private int getQueueCapacity(BlockingQueue<?> queue) {
        try {
            if (queue instanceof LinkedBlockingQueue) {
                Field field = LinkedBlockingQueue.class.getDeclaredField("capacity");
                field.setAccessible(true);
                return (Integer)field.get(queue);
            }
            if (queue instanceof ArrayBlockingQueue) {
                Field field = ArrayBlockingQueue.class.getDeclaredField("items");
                field.setAccessible(true);
                Object[] items = (Object[])field.get(queue);
                return items.length;
            }
        }
        catch (Exception e) {
            log.warn("\u83b7\u53d6\u961f\u5217\u5bb9\u91cf\u5931\u8d25", (Throwable)e);
        }
        return Integer.MAX_VALUE;
    }

    public Map<String, MonitoredThreadPool> getThreadPoolMap() {
        return this.threadPoolMap;
    }

    public Map<String, List<ThreadPoolStats>> getHistoryStats() {
        return this.historyStats;
    }

    public ThreadPoolMonitorProperties getMonitorProperties() {
        return this.monitorProperties;
    }

    public AlertService getAlertService() {
        return this.alertService;
    }
}

