package cn.longky.common.utils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;

/**
 * 模板渲染解析工具
 * <p>{@code TemplateUtils.format("${a} - ${b} = 1", 3, 2)} return "3 - 2 = 1"
 * <p>{@code TemplateUtils.format("${a} - ${a-1} = 1", 3)} return "3 - 2 = 1"
 * <p>{@code TemplateUtils.format("${a} - ${a-30} = 30", LocalDateTime.now())} return "20221025 - 20220925 = 30"
 *
 * @author yingzhan
 * @since 1.0
 */
public class TemplateUtils {

    public static String PLACEHOLDER_OPEN = "${";

    public static String PLACEHOLDER_CLOSE = "}";

    public static String format(String s, Object... args) {
        if (args.length == 0) {
            return s;
        }

        return format(s, new TemplateArgMap() {
            private int index = 0;

            @Override
            public boolean contains(String key) {
                return index < args.length;
            }

            @Override
            public Object get(String key) {
                return args[index++];
            }
        });
    }

    public static String format(String s, Map<String, Object> args) {
        if (args.isEmpty()) {
            return s;
        }

        return format(s, new TemplateArgMap() {
            @Override
            public boolean contains(String key) {
                return args.containsKey(key);
            }

            @Override
            public Object get(String key) {
                return args.get(key);
            }
        });
    }

    public static String format(String s, TemplateArgMap args) {
        if (s == null || args == null) {
            return s;
        }

        Map<String, Object> dict = new HashMap<>();
        int i = 0;
        StringBuilder sb = new StringBuilder();
        while (i < s.length()) {
            int startIndex = s.indexOf(PLACEHOLDER_OPEN, i);
            if (startIndex < 0) {
                sb.append(s, i, s.length());
                break;
            }

            int endIndex = s.indexOf(PLACEHOLDER_CLOSE, i + PLACEHOLDER_OPEN.length() + 1);
            if (endIndex < 0) {
                sb.append(s, i, s.length());
                break;
            }

            sb.append(s, i, startIndex);

            String placeholder = s.substring(startIndex + PLACEHOLDER_OPEN.length(), endIndex);
            String symbols = "+-";
            MAIN: {
                Object arg;
                for (char symbol : symbols.toCharArray()) {
                    int pos = placeholder.indexOf(symbol);
                    if (pos > 0 && pos + 1 < placeholder.length()) {
                        String argKey = placeholder.substring(0, pos);
                        if (dict.containsKey(argKey)) {
                            arg = dict.get(argKey);
                        } else if (args.contains(argKey)) {
                            arg = args.get(argKey);
                            dict.put(argKey, arg);
                        } else {
                            sb.append(s, startIndex, endIndex + 1);
                            break MAIN;
                        }

                        sb.append(dynamicRender(arg, symbol, placeholder.substring(pos + 1)));
                        break MAIN;
                    }
                }

                if (dict.containsKey(placeholder)) {
                    arg = dict.get(placeholder);
                } else if (args.contains(placeholder)) {
                    arg = args.get(placeholder);
                    dict.put(placeholder, arg);
                } else {
                    sb.append(s, startIndex, endIndex + 1);
                    break MAIN;
                }

                sb.append(dynamicRender(arg));
            }

            i = endIndex + 1;
        }
        return sb.toString();
    }

    /**
     * 模板动态渲染逻辑
     *
     * @param arg 参数值
     * @param op 操作符
     * @param suffix 动态参数后缀
     * @return 字符串渲染结果
     */
    private static Object dynamicRender(Object arg, char op, String suffix) {
        switch (op) {
            case '+' -> {
                if (arg instanceof Integer) {
                    return (int) arg + Integer.parseInt(suffix);
                } else if (arg instanceof Long) {
                    return (long) arg + Long.parseLong(suffix);
                } else if (arg instanceof LocalDate) {
                    LocalDate date = ((LocalDate) arg).plusDays(Integer.parseInt(suffix));
                    return DateTimeUtils.getVersionDaily(date);
                } else if (arg instanceof LocalDateTime) {
                    LocalDateTime dateTime = ((LocalDateTime) arg).plusDays(Integer.parseInt(suffix));
                    return DateTimeUtils.getVersionDaily(dateTime);
                }
            }
            case '-' -> {
                if (arg instanceof Integer) {
                    return (int) arg - Integer.parseInt(suffix);
                } else if (arg instanceof Long) {
                    return (long) arg - Long.parseLong(suffix);
                } else if (arg instanceof LocalDate) {
                    LocalDate date = ((LocalDate) arg).minusDays(Integer.parseInt(suffix));
                    return DateTimeUtils.getVersionDaily(date);
                } else if (arg instanceof LocalDateTime) {
                    LocalDateTime dateTime = ((LocalDateTime) arg).minusDays(Integer.parseInt(suffix));
                    return DateTimeUtils.getVersionDaily(dateTime);
                }
            }
            case '*' -> {
                if (arg instanceof Integer) {
                    return (int) arg * Integer.parseInt(suffix);
                } else if (arg instanceof Long) {
                    return (long) arg * Integer.parseInt(suffix);
                } else if (arg instanceof String) {
                    return ((String) arg).repeat(Integer.parseInt(suffix));
                }
            }
            default -> {
            }
        }
        return arg;
    }

    private static Object dynamicRender(Object arg) {
        if (arg instanceof LocalDate) {
            return DateTimeUtils.getVersionDaily((LocalDate) arg);
        } else if (arg instanceof LocalDateTime) {
            return DateTimeUtils.getVersionDaily((LocalDateTime) arg);
        }
        return arg;
    }

    public interface TemplateArgMap {
        boolean contains(String key);
        Object get(String key);
    }
}