package cn.longky.common.utils;

import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * @author yingzhan
 * @since 1.0
 */
public class DateTimeUtils {

    public static Duration durationOf(LocalDateTime time) {
        if (time == null) {
            return Duration.ZERO;
        }

        return Duration.ofMillis(LocalDateTime.now().until(time, ChronoUnit.MILLIS));
    }

    public static LocalDateTime asLocalDateTime(Date date) {
        if (date == null) {
            return null;
        }
        return asLocalDateTime(date.getTime());
    }

    public static LocalDateTime asLocalDateTime(Long tsMills) {
        if (tsMills == null) {
            return null;
        }
        return Instant.ofEpochMilli(tsMills).atZone(ZoneId.systemDefault()).toLocalDateTime();
    }

    public static LocalDate asLocalDate(Date date) {
        if (date == null) {
            return null;
        }
        return asLocalDate(date.getTime());
    }

    public static LocalDate asLocalDate(Long tsMills) {
        if (tsMills == null) {
            return null;
        }
        return Instant.ofEpochMilli(tsMills).atZone(ZoneId.systemDefault()).toLocalDate();
    }

    public static Date asDate(LocalDate localDate) {
        if (localDate == null) {
            return null;
        }
        return Date.from(localDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant());
    }

    public static Date asDate(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    public static Long asTimestamp(LocalDate localDate) {
        if (localDate == null) {
            return null;
        }
        return localDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    public static Long asTimestamp(LocalDateTime localDateTime) {
        if (localDateTime == null) {
            return null;
        }
        return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    public static Long asTimestamp(Date date) {
        if (date == null) {
            return null;
        }
        return date.getTime();
    }

    public static Instant asInstant(long timestamp) {
        if (timestamp == 0) {
            return null;
        }
        return Instant.ofEpochMilli(timestamp);
    }

    public static Instant asInstant(LocalDate date) {
        if (date == null) {
            return null;
        }
        return date.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant();
    }

    public static Instant asInstant(LocalDateTime dateTime) {
        if (dateTime == null) {
            return null;
        }
        return dateTime.atZone(ZoneId.systemDefault()).toInstant();
    }

    public static Instant asInstant(Date date) {
        if (date == null) {
            return null;
        }
        return date.toInstant();
    }

    public static int getSecondsByDay(int days) {
        if (days <= 0) {
            return 0;
        }

        long seconds = 3600L * 24 * days;
        if ((int) seconds != seconds) {
            throw new IllegalArgumentException("cannot get seconds, exceed maximum days, current days is " + days);
        }

        return (int) seconds;
    }

    public static long getCurrentVersionDaily() {
        return getVersionDaily(LocalDateTime.now());
    }

    public static long getVersionDaily(LocalDateTime dateTime) {
        return getVersionDaily(dateTime.toLocalDate());
    }

    public static long getVersionDaily(LocalDate date) {
        return date.getYear() * 100_00L
                + date.getMonthValue() * 100L
                + date.getDayOfMonth();
    }

    public static long getCurrentVersionTimestamp() {
        return getVersionTimestamp(LocalDateTime.now());
    }

    public static long getVersionTimestamp(LocalDateTime dateTime) {
        return getVersionDaily(dateTime) * 100_00_00
                + dateTime.getHour() * 100_00
                + dateTime.getMinute() * 100
                + dateTime.getSecond();
    }

    /**
     * 序列化时间
     *
     * <p>e.g. {@code stringify(3601002, TimeUnit.HOURS, TimeUnit.MILLS);} return "1时1秒2毫秒"
     * <p>e.g. {@code stringify(3601002, TimeUnit.HOURS, TimeUnit.SECONDS);} return "1时1秒"
     * <p>e.g. {@code stringify(3601002, TimeUnit.MINUTES, TimeUnit.SECONDS);} return "60分1秒"
     *
     * @param timeMills 毫秒数
     * @param upperPrecision 精度上限
     * @param lowerPrecision 精度下限
     * @return 序列化字符串
     */
    public static String stringify(long timeMills, TimeUnit upperPrecision, TimeUnit lowerPrecision) {
        if (timeMills <= 0) {
            return "0";
        }

        StringBuilder sb = new StringBuilder();
        if (timeMills >= 86400_000
                && (upperPrecision == null || upperPrecision.ordinal() >= TimeUnit.DAYS.ordinal())) {
            sb.append(timeMills / 86400_000).append("天");
            timeMills %= 86400_000;
        }
        if (timeMills >= 3600_000
                && (upperPrecision == null || upperPrecision.ordinal() >= TimeUnit.HOURS.ordinal())
                && (lowerPrecision == null || lowerPrecision.ordinal() <= TimeUnit.HOURS.ordinal())) {
            sb.append(timeMills / 3600_000).append("时");
            timeMills %= 3600_000;
        }
        if (timeMills >= 60_000
                && (upperPrecision == null || upperPrecision.ordinal() >= TimeUnit.MINUTES.ordinal())
                && (lowerPrecision == null || lowerPrecision.ordinal() <= TimeUnit.MINUTES.ordinal())) {
            sb.append(timeMills / 60_000).append("分");
            timeMills %= 60_000;
        }
        if (timeMills >= 1_000
                && (upperPrecision == null || upperPrecision.ordinal() >= TimeUnit.SECONDS.ordinal())
                && (lowerPrecision == null || lowerPrecision.ordinal() <= TimeUnit.SECONDS.ordinal())) {
            sb.append(timeMills / 1_000).append("秒");
            timeMills %= 1_000;
        }
        if (timeMills > 0
                && (upperPrecision == null || upperPrecision.ordinal() >= TimeUnit.MILLISECONDS.ordinal())
                && (lowerPrecision == null || lowerPrecision.ordinal() <= TimeUnit.MILLISECONDS.ordinal())) {
            sb.append(timeMills).append("毫秒");
        }
        return sb.toString();
    }

    public static String stringify(Duration duration, TimeUnit upperPrecision, TimeUnit lowerPrecision) {
        return stringify(duration.toMillis(), upperPrecision, lowerPrecision);
    }

    public static String stringify(long timeMills) {
        return stringify(timeMills, null, null);
    }

    public static String format(LocalDateTime dateTime, String pattern) {
        if (dateTime == null) {
            return null;
        }

        DateTimeFormatter formatter;
        if (pattern != null) {
            formatter = DateTimeFormatter.ofPattern(pattern);
        } else {
            formatter = DateTimeFormatter.ISO_DATE_TIME;
        }

        return dateTime.format(formatter);
    }

    public static String simpleFormat(LocalDateTime dateTime) {
        return format(dateTime, "yyyy-MM-dd HH:mm:ss");
    }

    private static final DateTimeFormatter DTF_RFC_3399 = DateTimeFormatter
            .ofPattern("yyyy-MM-dd'T'HH:mm:ss+08:00")
            .withZone(ZoneId.of("Asia/Shanghai"));
    public static String rfc3339Format(LocalDateTime dateTime) {
        if (dateTime == null) {
            return null;
        }

        return dateTime.format(DTF_RFC_3399);
    }
}
