package cn.longky.common.utils;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * @author yingzhan
 * @since 1.0
 */
public class ConvertUtils {

    public static <T extends Enum<T>> T convert(String name, Class<T> clazz) {
        if (name == null) {
            return null;
        }

        try {
            return T.valueOf(clazz, name);
        } catch (Exception ignored) {
            return null;
        }
    }

    public static <T extends Enum<T>> T convert(Integer ordinal, Class<T> clazz) {
        if (ordinal == null) {
            return null;
        }

        T[] enums = clazz.getEnumConstants();
        if (ordinal >= 0 && ordinal < enums.length) {
            return enums[ordinal];
        }

        return null;
    }

    public static <S, T extends Enum<T>> T convert(S source, Class<T> clazz, Function<T, S> sourceGetter) {
        if (source == null) {
            return null;
        }

        for (T enumConstant : clazz.getEnumConstants()) {
            if (Objects.equals(sourceGetter.apply(enumConstant), source)) {
                return enumConstant;
            }
        }

        return null;
    }

    public static <T extends Enum<T>> Integer convert2Int(String name, Class<T> clazz) {
        T t = convert(name, clazz);
        return t != null ? t.ordinal() : null;
    }

    public static <T extends Enum<T>> String convert2String(Integer ordinal, Class<T> clazz) {
        T t = convert(ordinal, clazz);
        return t != null ? t.name() : null;
    }

    public static <S, T> List<T> convert(Collection<S> sources, Function<S, T> converter) {
        if (sources == null || converter == null) {
            return null;
        }

        return sources.stream().map(converter).filter(Objects::nonNull).collect(Collectors.toList());
    }

    /**
     * 集合类型转换
     *
     * @param t 入参
     * @param initializer 出参初始化器
     * @param converter 元素转换器
     * @return 集合类型结果
     * @param <U> 入参元素类型
     * @param <V> 出参元素类型
     * @param <T> 入参集合类型
     * @param <R> 出参集合类型
     */
    public static  <U, V, T extends Collection<U>, R extends Collection<V>> R convert(
            T t, Supplier<R> initializer, Function<U, V> converter) {
        if (t == null) {
            return null;
        }

        R r = initializer.get();
        for (U u : t) {
            V v = null;
            try {
                v = converter.apply(u);
            } finally {
                if (v != null) {
                    r.add(v);
                }
            }
        }
        return r;
    }
}
