package cn.longky.common.utils;

import cn.longky.common.function.KRunnable;
import cn.longky.common.function.KSupplier;
import cn.longky.common.model.KException;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.RandomStringUtils;

import java.time.Duration;
import java.util.Objects;
import java.util.Random;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

/**
 * @author yingzhan
 * @since 1.0
 */
public class CommonUtils {

    private static final Random RANDOM = new Random();

    @SuppressWarnings("unchecked")
    public static <T> T safeCast(Object o, Class<T> clazz) {
        if (clazz.isInstance(o)) {
            return (T) o;
        }
        if (clazz == Long.class) {
            if (o instanceof Integer) return clazz.cast((long) (int) o);
            if (o instanceof Byte) return clazz.cast((long) (byte) o);
            if (o instanceof Character) return clazz.cast((long) (char) o);
            if (o instanceof Double) return clazz.cast((long) (double) o);
            if (o instanceof Float) return clazz.cast((long) (float) o);
        } else if (clazz == Integer.class) {
            if (o instanceof Long) return clazz.cast((int) (long) o);
            if (o instanceof Byte) return clazz.cast((int) (byte) o);
            if (o instanceof Character) return clazz.cast((int) (char) o);
            if (o instanceof Double) return clazz.cast((int) (double) o);
            if (o instanceof Float) return clazz.cast((int) (float) o);
        }
        return null;
    }

    @SafeVarargs
    public static <T> T firstNonnull(T... args) {
        for (T t : args) {
            if (t != null) {
                return t;
            }
        }

        throw new IllegalArgumentException("Arguments cannot be all nullable.");
    }

    @SafeVarargs
    public static <T> T firstNonnull(T defaultValue, Supplier<T>... suppliers) {
        for (Supplier<T> supplier : suppliers) {
            if (supplier == null) {
                continue;
            }

            T t = supplier.get();
            if (t != null) {
                return t;
            }
        }
        return defaultValue;
    }

    @SafeVarargs
    public static <T> T firstMatch(Function<T, Boolean> matcher, T... args) {
        for (T t : args) {
            if (matcher.apply(t)) {
                return t;
            }
        }
        return null;
    }

    @SafeVarargs
    public static <T> T firstMatchWithSuppliers(Function<T, Boolean> matcher, Supplier<T>... suppliers) {
        for (Supplier<T> supplier : suppliers) {
            if (supplier == null) {
                continue;
            }

            T t = supplier.get();
            if (matcher.apply(t)) {
                return t;
            }
        }

        return null;
    }

    public static <T> T swallowException(KSupplier<T> supplier) {
        try {
            return supplier.get();
        } catch (Exception ignored) {
        }
        return null;
    }

    public static void swallowException(KRunnable runnable) {
        try {
            runnable.run();
        } catch (KException ignored) {
        }
    }

    public static String getUuid() {
        return getUuid("");
    }

    public static String getUuid(String replacement) {
        String uuid = UUID.randomUUID().toString();
        if (replacement == null) {
            return uuid;
        } else {
            return uuid.replace("-", replacement);
        }
    }

    @SafeVarargs
    public static <T> boolean contains(T item, T... array) {
        if (array.length == 0 || item == null) {
            return false;
        }

        for (T t : array) {
            if (Objects.equals(t, item)) {
                return true;
            }
        }
        return false;
    }

    public static String encryptPassword(String password, String salt) {
        String raw = password + salt;
        return DigestUtils.sha256Hex(raw);
    }

    public static boolean validatePassword(String rawPassword, String encryptedPassword, String salt) {
        if (StringUtils.isBlank(rawPassword)
                || StringUtils.isBlank(encryptedPassword)
                || StringUtils.isBlank(salt)) {
            return false;
        }

        return encryptedPassword.equals(encryptPassword(rawPassword, salt));
    }

    private static final char[] CHARS = {
            '1', '2', '3', '4', '5', '6', '7', '8', '9', // 9
            'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'j', 'k',
            'm', 'n', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w',
            'x', 'y', 'z', // 9 + 23 = 32
            'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'J', 'K',
            'M', 'N', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W',
            'X', 'Y', 'Z', // 9 + 23 + 23 = 55
    };

    public static String generateRandomString(int length, boolean isNumeric, boolean ignoreCase) {
        if (length <= 0) {
            throw new IllegalArgumentException("生成的字符串长度必须是正整数");
        }

        if (isNumeric) {
            return RandomStringUtils.randomNumeric(length);
        }

        final StringBuilder sb = new StringBuilder(length);
        int charLength = ignoreCase ? 32 : CHARS.length;
        while (length-- > 0) {
            int idx = RANDOM.nextInt(charLength);
            sb.append(CHARS[idx]);
        }
        return sb.toString();
    }


    public static String generateUniqueCodeByUid(long userId, int length, boolean isNumeric,
                                                 int primeOfCharLength, long salt) {
        if (length < 4) {
            throw new IllegalArgumentException("所生成的目标码长度不能少于4");
        }
        long uid = userId * primeOfCharLength + salt;
        char[] codes = new char[length];
        int[] ids = new int[length];
        int charLength = isNumeric ? 9 : CHARS.length;
        for (int i = 0; i < length; ++i) {
            ids[i] = (int) (uid % charLength);
            ids[i] = (ids[i] + i * ids[0]) % charLength;
            uid /= charLength;
        }

        for (int i = 0; i < length; ++i) {
            int idx = (i * (length - 1)) % length;
            codes[i] = CHARS[ids[idx]];
        }
        return new String(codes);
    }

    public static String generateUniqueCodeByUid(long userId, int length, boolean isNumeric) {
        return generateUniqueCodeByUid(userId, length, isNumeric, 23, 1_2345_6789);
    }

    public static String stringifySize(long size) {
        if (size < 0) {
            throw new IllegalArgumentException("尺寸大小不能小于0");
        }

        StringBuilder sb = new StringBuilder();
        if (size >= 1024L * 1024 * 1024 * 1024 * 1024) {
            sb.append(size / (1024L * 1024 * 1024 * 1024 * 1024)).append("PB");
            size %= (1024L * 1024 * 1024 * 1024 * 1024);
        }
        if (size >= 1024L * 1024 * 1024 * 1024) {
            sb.append(size / (1024L * 1024 * 1024 * 1024)).append("TB");
            size %= (1024L * 1024 * 1024 * 1024);
        }
        if (size >= 1024L * 1024 * 1024) {
            sb.append(size / (1024L * 1024 * 1024)).append("GB");
            size %= (1024L * 1024 * 1024);
        }
        if (size >= 1024L * 1024) {
            sb.append(size / (1024L * 1024)).append("MB");
            size %= (1024L * 1024);
        }
        if (size >= 1024L) {
            sb.append(size / 1024L).append("KB");
            size %= 1024L;
        }
        if (size > 0) {
            sb.append(size).append("B");
        }
        return sb.toString();
    }

    public static <T> T retry(KSupplier<T> supplier,
                              int maxRetries) throws Throwable {
        return retry(supplier, maxRetries, null, null, null);
    }

    public static <T> T retry(KSupplier<T> supplier,
                              int maxRetries,
                              Duration sleepTime,
                              Class<? extends Throwable> throwableClass,
                              Predicate<T> verifier) throws Throwable {
        return retry(supplier, maxRetries, sleepTime, throwableClass, verifier, null);
    }

    public static <T> T retry(KSupplier<T> supplier,
                              int maxRetries,
                              Duration sleepTime,
                              Class<? extends Throwable> throwableClass,
                              Predicate<T> verifier,
                              T defaultValue) throws Throwable {
        while (maxRetries-- > 0) {
            try {
                T t = supplier.get();
                if (maxRetries <= 0 || verifier == null || verifier.test(t)) {
                    return t;
                }
            } catch (Throwable t) {
                if (maxRetries <= 0) {
                    throw t;
                }
                if (throwableClass != null && throwableClass.isInstance(t)) {
                    throw t;
                }
            }

            if (sleepTime != null) {
                Thread.sleep(sleepTime.toMillis());
            }
        }

        return defaultValue;
    }

    public static Long parseLong(Object o) {
        if (o == null) {
            return null;
        }
        if (o instanceof Long) {
            return (Long) o;
        }
        if (o instanceof Number) {
            return ((Number) o).longValue();
        }
        if (o instanceof Character) {
            return (long) (Character) o;
        }
        if (o instanceof CharSequence) {
            return Long.parseLong(((CharSequence) o).toString());
        }
        throw new NumberFormatException();
    }

    public static Integer parseInt(Object o) {
        if (o == null) {
            return null;
        }
        if (o instanceof Integer) {
            return (Integer) o;
        }
        if (o instanceof Number) {
            return ((Number) o).intValue();
        }
        if (o instanceof Character) {
            return (int) (Character) o;
        }
        if (o instanceof CharSequence) {
            return Integer.parseInt(((CharSequence) o).toString());
        }
        throw new NumberFormatException();
    }

    public static Boolean parseBoolean(Object o) {
        if (o == null) {
            return null;
        }
        if (o instanceof Boolean) {
            return (Boolean) o;
        }
        if (o instanceof Number) {
            return ((Number) o).intValue() != 0;
        }
        if (o instanceof CharSequence) {
            String s = ((CharSequence) o).toString();
            return "true".equalsIgnoreCase(s) || "1".equals(s);
        }
        throw new NumberFormatException();
    }
}