package cn.longky.common.model;

import lombok.Data;

/**
 * 请求结果
 *
 * @author yingzhan
 * @since 1.0
 */
@Data
public class Result<T> {

    /**
     * 结果是否成功
     */
    private boolean success;

    /**
     * 如果success = true，返回结果值
     * 如果success = false，返回null
     */
    private T data;

    /**
     * 错误码
     */
    private String errorCode;

    /**
     * 错误信息
     */
    private String errorMessage;

    protected Result() {
    }

    protected Result(boolean success) {
        this.success = success;
    }

    /**
     * 构造请求错误结果
     *
     * @param errorCode 错误码
     * @return 请求结果
     */
    public Result<T> withError(ErrorCode errorCode) {
        return withError(errorCode.getCode(), errorCode.getMessage());
    }

    /**
     * 构造请求异常结果
     *
     * @param throwable 异常信息
     * @return 请求结果
     */
    public Result<T> withThrowable(Throwable throwable) {
        KException exception = KException.tryWrapUnknown(throwable);
        return withError(exception.getErrorCode(), exception.getMessage());
    }

    /**
     * 构造请求结果
     *
     * @param errorCode 错误码
     * @param errorMsg 错误信息
     * @return 请求结果
     */
    public Result<T> withError(String errorCode, String errorMsg) {
        setSuccess(false);
        setErrorCode(errorCode);
        setErrorMessage(errorMsg);
        return this;
    }

    /**
     * 构造请求结果
     *
     * @param result 结果数据
     * @return 请求结果
     */
    public Result<T> withSuccess(T result) {
        setSuccess(true);
        setData(result);
        return this;
    }

    /**
     * 构造请求结果
     *
     * @param result 结果数据
     * @return 请求结果
     * @param <T> 数据类型
     */
    public static <T> Result<T> ofSuccess(T result) {
        return (new Result<T>()).withSuccess(result);
    }

    /**
     * 构造请求结果
     *
     * @param errorCode 错误码对象
     * @return 请求结果
     * @param <T> 数据类型
     */
    public static <T> Result<T> ofError(ErrorCode errorCode) {
        return (new Result<T>()).withError(errorCode);
    }

    /**
     * 构造请求结果
     *
     * @param errorCode 错误码
     * @param errorMsg 错误信息
     * @return 请求结果
     * @param <T> 数据类型
     */
    public static <T> Result<T> ofError(String errorCode, String errorMsg) {
        return (new Result<T>()).withError(errorCode, errorMsg);
    }

    /**
     * 构造请求结果
     *
     * @param t 异常信息
     * @return 请求结果
     * @param <T> 数据类型
     */
    public static <T> Result<T> ofThrowable(Throwable t) {
        return (new Result<T>()).withThrowable(t);
    }
}
