package cn.lonelysnow.common.utils.json;

import cn.lonelysnow.common.exception.CommonExceptionEnum;
import cn.lonelysnow.common.exception.SnowException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;

/**
 * @author LonelySnow
 * @classname JsonUtil
 * @description Json解析工具类
 * @created LonelySnow
 * @date 2021/8/7 11:35 上午
 */
@Slf4j
public class JacksonUtils {

    public static final ObjectMapper mapper = new ObjectMapper();

    /**
     * 内容转json
     * @author LonelySnow
     * @param obj
     * @result java.lang.String
     * @date 2021/8/7 11:42 上午
     */
    public static String toString(Object obj) {
        if (obj == null) {
            return null;
        }
        if (obj.getClass() == String.class) {
            return (String) obj;
        }
        try {
            return mapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            log.error(MessageFormat.format("Json序列化出错，错误内容{0}", e));
            throw new SnowException(CommonExceptionEnum.JSON_PARSING_ERROR);
        }
    }

    /**
     * json转实体类
     * @author LonelySnow
     * @param json
     * @param tClass
     * @result T
     * @date 2021/8/7 11:44 上午
     */
    public static <T> T toBean(String json, Class<T> tClass) {
        try {
            return mapper.readValue(json, tClass);
        } catch (Exception e) {
            log.error(MessageFormat.format("Json解析出错，错误内容{0}", e));
            throw new SnowException(CommonExceptionEnum.JSON_PARSING_ERROR);
        }
    }

    /**
     * json转List
     * @author LonelySnow
     * @param json
     * @param tClass
     * @result T
     * @date 2021/8/7 11:46 上午
     */
    public static <T> T toList(String json, Class<T> tClass) {
        try {
            return mapper.readValue(json, mapper.getTypeFactory().constructCollectionType(List.class, tClass));
        } catch (Exception e) {
            log.error(MessageFormat.format("Json解析出错，错误内容{0}", e));
            throw new SnowException(CommonExceptionEnum.JSON_PARSING_ERROR);
        }
    }

    /**
     * json转Map
     * @author LonelySnow
     * @param json
     * @param kClass
     * @param vClass
     * @result java.util.Map<K,V>
     * @date 2021/8/7 11:49 上午
     */
    public static <K, V> Map<K, V> toMap(String json, Class<K> kClass, Class<V> vClass) {
        try {
            return mapper.readValue(json, mapper.getTypeFactory().constructMapType(Map.class, kClass, vClass));
        } catch (Exception e) {
            log.error(MessageFormat.format("Json解析出错，错误内容{0}", e));
            throw new SnowException(CommonExceptionEnum.JSON_PARSING_ERROR);
        }
    }

}
