package cn.lonelysnow.common.utils.ip;

import cn.lonelysnow.common.utils.call.ApiCallUtils;
import com.alibaba.fastjson2.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.HttpServletRequest;
import java.util.Objects;

/**
 * @author LonelySnow
 * @classname IpUtils
 * @description IP指向地址获取
 * @created LonelySnow
 * @date 2022/4/15 13:21
 */
@Slf4j
public class IpUtils {

    /**
     * 设置本机地址
     * @author LonelySnow
     * @date 2022/4/15 13:21
     */
    private static final String LOCAL_IP = "127.0.0.1";

    /**
     * 未知地址
     * @author LonelySnow
     * @date 2022/4/15 13:22
     */
    public static final String UNKNOWN = "XX XX";

    /**
     * IP地址查询接口
     * @author LonelySnow
     * @date 2022/4/15 13:22
     */
    public static final String IP_QUERY_URL = "https://whois.pconline.com.cn/ipJson.jsp?ip=%s&json=true";

    /**
     * 获取客户端ip
     * 获取客户端的IP地址的方法是：request.getRemoteAddr()，这种方法在大部分情况下都是有效的。
     * 但是在通过了Apache, Squid等反向代理软件就不能获取到客户端的真实IP地址了，如果通过了多级反向代理的话，
     * X-Forwarded-For的值并不止一个，而是一串IP值，究竟哪个才是真正的用户端的真实IP呢？
     * 答案是取X-Forwarded-For中第一个非unknown的有效IP字符串。
     * 例如：X-Forwarded-For：192.168.1.110, 192.168.1.120, 192.168.1.130, 192.168.1.100
     * 用户真实IP为：192.168.1.110
     * @author LonelySnow
     * @date 2022/4/15 13:22
     */
    public static String getIpAddr(HttpServletRequest request) {
        String unknown = "unknown";
        if (request == null) {
            return unknown;
        }
        String ip = request.getHeader("X-Real-IP");
        if (StringUtils.isBlank(ip) || unknown.equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Forwarded-For");
        }
        if (StringUtils.isBlank(ip) || unknown.equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (StringUtils.isBlank(ip) || unknown.equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (StringUtils.isBlank(ip) || unknown.equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        // 对于通过多个代理的情况, 第一个IP为客户端真实IP, 多个IP按照','分割, "***.***.***.***".length() = 15
        if (ip != null && ip.length() > 15) {
            if (ip.indexOf(",") > 0) {
                ip = ip.substring(0, ip.indexOf(","));
            }
        }
        return "0:0:0:0:0:0:0:1".equals(ip) ? LOCAL_IP : ip;
    }

    /**
     * 根据IP地址查询地理位置
     * @author LonelySnow
     * @date 2022/4/15 13:23
     */
    public static String getLocationByIp(String ip) {
        // 内网不查询
        if (internalIp(ip)) {
            return "内网IP";
        }
        try {
            // 请求地址，获取结果
            String rspStr = ApiCallUtils.doGetResJson(String.format(IP_QUERY_URL, ip));
            if (StringUtils.isEmpty(rspStr)) {
                log.error("获取地理位置异常：{}", ip);
                return UNKNOWN;
            }
            JSONObject obj = JSONObject.parseObject(rspStr);
            String region = obj.getString("pro");
            String city = obj.getString("city");
            String addr = obj.getString("addr");
            if (StringUtils.isEmpty(region) && StringUtils.isEmpty(city)) {
                return StringUtils.isNotBlank(addr) ? addr.trim() : UNKNOWN;
            }
            return String.format("%s %s", region, city);
        } catch (Exception e) {
            log.error("获取地理位置异常", e);
        }
        return UNKNOWN;
    }

    /**
     * 判断IP地址是不是内网IP
     * @author LonelySnow
     * @date 2022/4/15 13:23
     */
    public static boolean internalIp(String ip) {
        byte[] ipBytes = textToNumericFormatV4(ip);
        return isLocalIp(ip) || internalIp(ipBytes);
    }

    /**
     * 将IPv4地址转换成字节
     * @author LonelySnow
     * @date 2022/4/15 13:23
     */
    public static byte[] textToNumericFormatV4(String text) {
        if (text.length() == 0) {
            return null;
        }
        byte[] bytes = new byte[4];
        String[] elements = text.split("\\.", -1);
        try {
            long l;
            int i;
            switch (elements.length) {
                case 1:
                    l = Long.parseLong(elements[0]);
                    if ((l < 0L) || (l > 4294967295L)) {
                        return null;
                    }
                    bytes[0] = (byte) (int) (l >> 24 & 0xFF);
                    bytes[1] = (byte) (int) ((l & 0xFFFFFF) >> 16 & 0xFF);
                    bytes[2] = (byte) (int) ((l & 0xFFFF) >> 8 & 0xFF);
                    bytes[3] = (byte) (int) (l & 0xFF);
                    break;
                case 2:
                    l = Integer.parseInt(elements[0]);
                    if ((l < 0L) || (l > 255L)) {
                        return null;
                    }
                    bytes[0] = (byte) (int) (l & 0xFF);
                    l = Integer.parseInt(elements[1]);
                    if ((l < 0L) || (l > 16777215L)) {
                        return null;
                    }
                    bytes[1] = (byte) (int) (l >> 16 & 0xFF);
                    bytes[2] = (byte) (int) ((l & 0xFFFF) >> 8 & 0xFF);
                    bytes[3] = (byte) (int) (l & 0xFF);
                    break;
                case 3:
                    for (i = 0; i < 2; ++i) {
                        l = Integer.parseInt(elements[i]);
                        if ((l < 0L) || (l > 255L)) {
                            return null;
                        }
                        bytes[i] = (byte) (int) (l & 0xFF);
                    }
                    l = Integer.parseInt(elements[2]);
                    if ((l < 0L) || (l > 65535L)) {
                        return null;
                    }
                    bytes[2] = (byte) (int) (l >> 8 & 0xFF);
                    bytes[3] = (byte) (int) (l & 0xFF);
                    break;
                case 4:
                    for (i = 0; i < 4; ++i) {
                        l = Integer.parseInt(elements[i]);
                        if ((l < 0L) || (l > 255L)) {
                            return null;
                        }
                        bytes[i] = (byte) (int) (l & 0xFF);
                    }
                    break;
                default:
                    return null;
            }
        } catch (NumberFormatException ignored) {
            return null;
        }
        return bytes;
    }

    /**
     * 判断IP地址是否为本地IP
     * @author LonelySnow
     * @date 2022/4/15 13:24
     */
    public static boolean isLocalIp(String ip) {
        return LOCAL_IP.equals(ip) || "0:0:0:0:0:0:0:1".equals(ip);
    }

    /**
     *
     * @author LonelySnow
     * @date 2022/4/15 13:24
     */
    private static boolean internalIp(byte[] ipBytes) {
        if (Objects.isNull(ipBytes) || ipBytes.length < 2) {
            return true;
        }
        final byte b0 = ipBytes[0];
        final byte b1 = ipBytes[1];
        // 10.x.x.x/8
        final byte section1 = 0x0A;
        // 172.16.x.x/12
        final byte section2 = (byte) 0xAC;
        final byte section3 = (byte) 0x10;
        final byte section4 = (byte) 0x1F;
        // 192.168.x.x/16
        final byte section5 = (byte) 0xC0;
        final byte section6 = (byte) 0xA8;
        switch (b0) {
            case section1:
                return true;
            case section2:
                if (b1 >= section3 && b1 <= section4) {
                    return true;
                }
            case section5:
                if (b1 == section6) {
                    return true;
                }
            default:
                return false;
        }
    }

    private IpUtils() {
    }

}
