package cn.lonelysnow.common.utils.call;

import cn.lonelysnow.common.exception.CommonExceptionEnum;
import cn.lonelysnow.common.exception.SnowException;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.springframework.http.*;
import org.springframework.web.client.RestTemplate;

/**
 * 进行第三方接口调用工具类
 * 传递参数一般为 String类型请求地址(url)、实体类类型所需传递参数(source,json类型内置转化方法，无序额外操作)、返回值类型实体类class对象(target)
 * 方法命名规则的：do + 请求类型 + 发送数据类型 + Res + 返回参数类型
 *  如：doPostPojoResJson表示：发送Post请求，传递参数为实体类类型参数，接收请求结果为json格式结果
 *
 * @author LonelySnow
 * @date 2020/10/12 19:22
 * @version V2.0
 */
public class ApiCallUtils {

    /**
     * 发送get请求，无参，json格式返回值
     */
    public static String doGetResJson(String url) {
        RestTemplate restTemplate = new RestTemplate();
        String result = restTemplate.getForObject(url, String.class);
        return result;
    }

    /**
     * 发送get请求，无参，实体类格式返回值
     */
    public static <T> T doGetResPojo(String url, Class<T> target) {
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> result = restTemplate.getForEntity(url, target);
        return result.getBody();
    }

    /**
     * 发送get请求，json格式参数，json格式返回值
     */
    public static <T> String doGetJsonResJson(String url, T source) {
        HttpEntity<String> jsonData = toJson(source);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<String> result = restTemplate.exchange(url, HttpMethod.GET, jsonData, String.class);
        return result.getBody();
    }

    /**
     * 发送get请求，json格式参数，实体类格式返回值
     */
    public static <T> T doGetJsonResPojo(String url, T source, Class<T> target) {
        HttpEntity<String> jsonData = toJson(source);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> result = restTemplate.exchange(url, HttpMethod.GET, jsonData, target);
        return result.getBody();
    }

    /**
     * 发送post请求，json格式参数，实体类格式返回值
     */
    public static <T> T doPostJsonResPojo(String url, T source, Class<T> target) {
        HttpEntity<String> jsonData = toJson(source);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> result = restTemplate.postForEntity(url, jsonData, target);
        return result.getBody();
    }

    /**
     * 发送post请求，json格式参数，json格式返回值
     */
    public static <T> String doPostJsonResJson(String url, T source) {
        HttpEntity<String> jsonData = toJson(source);
        RestTemplate restTemplate = new RestTemplate();
        String result = restTemplate.postForObject(url, jsonData, String.class);
        return result;
    }

    /**
     * 发送post请求，实体类格式参数，实体类格式返回值
     */
    public static <T> T doPostPojoResPojo(String url, T source, Class<T> target) {
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> result = restTemplate.postForEntity(url, source, target);
        return result.getBody();
    }

    /**
     * 发送post请求，实体类格式参数，json格式返回值
     */
    public static <T> String doPostPojoResJson(String url, T source) {
        RestTemplate restTemplate = new RestTemplate();
        String result = restTemplate.postForObject(url, source, String.class);
        return result;
    }

    /**
     * 发送put请求，json格式参数，实体类格式返回值
     */
    public static <T> T doPutJsonResPojo(String url, T source, Class<T> target) {
        HttpEntity<String> jsonData = toJson(source);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> result = restTemplate.exchange(url, HttpMethod.PUT, jsonData, target);
        return result.getBody();
    }

    /**
     * 发送put请求，json格式参数，json格式返回值
     */
    public static <T> String doPutJsonResJson(String url, T target) {
        HttpEntity<String> jsonData = toJson(target);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<String> result = restTemplate.exchange(url, HttpMethod.PUT, jsonData, String.class);
        return result.getBody();
    }

    /**
     * 发送delete请求，json格式参数，实体类格式返回值
     */
    public static <T> T doDeleteJsonResPojo(String url, T source, Class<T> target) {
        HttpEntity<String> jsonData = toJson(source);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<T> result = restTemplate.exchange(url, HttpMethod.DELETE, jsonData, target);
        return result.getBody();
    }

    /**
     * 发送delete请求，json格式参数，json格式返回值
     */
    public static <T> String doDeleteJsonResJson(String url, T target) {
        HttpEntity<String> jsonData = toJson(target);
        RestTemplate restTemplate = new RestTemplate();
        ResponseEntity<String> result = restTemplate.exchange(url, HttpMethod.DELETE, jsonData, String.class);
        return result.getBody();
    }

    /**
     * 发送json格式参数数据格式转换
     */
    private static <T> HttpEntity<String> toJson(T source) {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        String jsonStr = "";
        try {
            jsonStr = mapper.writeValueAsString(source);
        } catch (JsonProcessingException e) {
            throw new SnowException(CommonExceptionEnum.DATA_TRANSFER_ERROR);
        }
        HttpHeaders headers = new HttpHeaders();
        MediaType type = MediaType.parseMediaType("application/json; charset=UTF-8");
        headers.setContentType(type);
        headers.add("Accept", MediaType.APPLICATION_JSON.toString());
        HttpEntity<String> jsonData = new HttpEntity<>(jsonStr, headers);
        return jsonData;
    }

}
