package cn.lonelysnow.common.auth;

import cn.lonelysnow.common.exception.CommonExceptionEnum;
import cn.lonelysnow.common.exception.SnowException;
import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;

/**
 * RSA非对称加密工具类
 * @author lonelysnow
 */
@Slf4j
public class RsaUtils {

    private static final int DEFAULT_KEY_SIZE = 2048;

    /**
     * 从文件中读取公钥
     * @param filename 公钥保存路径，相对于classpath
     * @return 公钥对象
     * @throws Exception
     */
    public static PublicKey getPublicKey(String filename) {
        byte[] bytes = readFile(filename);
        return getPublicKey(bytes);
    }

    /**
     * 从文件中读取密钥
     * @param filename 私钥保存路径，相对于classpath
     * @return 私钥对象
     * @throws Exception
     */
    public static PrivateKey getPrivateKey(String filename) {
        byte[] bytes = readFile(filename);
        return getPrivateKey(bytes);
    }

    /**
     * 获取公钥
     * @param bytes 公钥的字节形式
     * @return
     * @throws Exception
     */
    private static PublicKey getPublicKey(byte[] bytes) {
        try {
            bytes = Base64.getDecoder().decode(bytes);
            X509EncodedKeySpec spec = new X509EncodedKeySpec(bytes);
            KeyFactory factory = KeyFactory.getInstance("RSA");
            return factory.generatePublic(spec);
        } catch (NoSuchAlgorithmException e) {
            log.error("算法异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_NO_SUCH_ALGORITHM);
        } catch (InvalidKeySpecException e) {
            log.error("秘钥失效：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_INVALID_KEY_SPEC);
        }
    }

    /**
     * 获取私钥
     * @param bytes 私钥的字节形式
     * @return
     * @throws Exception
     */
    private static PrivateKey getPrivateKey(byte[] bytes) {
        try {
            bytes = Base64.getDecoder().decode(bytes);
            PKCS8EncodedKeySpec spec = new PKCS8EncodedKeySpec(bytes);
            KeyFactory factory = KeyFactory.getInstance("RSA");
            return factory.generatePrivate(spec);
        } catch (NoSuchAlgorithmException e) {
            log.error("算法异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_NO_SUCH_ALGORITHM);
        } catch (InvalidKeySpecException e) {
            log.error("秘钥失效：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_INVALID_KEY_SPEC);
        }
    }

    /**
     * 根据密文，生存rsa公钥和私钥,并写入指定文件
     * @param publicKeyFilename  公钥文件路径
     * @param privateKeyFilename 私钥文件路径
     * @param secret             生成密钥的密文
     */
    public static void generateKey(String publicKeyFilename, String privateKeyFilename, String secret, int keySize) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance("RSA");
            SecureRandom secureRandom = new SecureRandom(secret.getBytes());
            keyPairGenerator.initialize(Math.max(keySize, DEFAULT_KEY_SIZE), secureRandom);
            KeyPair keyPair = keyPairGenerator.genKeyPair();
            // 获取公钥并写出
            byte[] publicKeyBytes = keyPair.getPublic().getEncoded();
            publicKeyBytes = Base64.getEncoder().encode(publicKeyBytes);
            writeFile(publicKeyFilename, publicKeyBytes);
            // 获取私钥并写出
            byte[] privateKeyBytes = keyPair.getPrivate().getEncoded();
            privateKeyBytes = Base64.getEncoder().encode(privateKeyBytes);
            writeFile(privateKeyFilename, privateKeyBytes);
        } catch (NoSuchAlgorithmException e) {
            log.error("生成秘钥，算法异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_NO_SUCH_ALGORITHM);
        }
    }

    /**
     * 读取文件
     * @author LonelySnow
     * @date 2022/9/13 09:18
     */
    private static byte[] readFile(String fileName) {
        try {
            return Files.readAllBytes(new File(fileName).toPath());
        } catch (IOException e) {
            log.error("RSA秘钥读取异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_FILE_READ_ERROR);
        }
    }

    /**
     * 写入文件
     * @author LonelySnow
     * @date 2022/9/13 09:19
     */
    private static void writeFile(String destPath, byte[] bytes) {
        try {
            File dest = new File(destPath);
            if (!dest.exists()) {
                dest.createNewFile();
            }
            Files.write(dest.toPath(), bytes);
        } catch (IOException e) {
            log.error("RSA秘钥文件创建或写入异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.RSA_FILE_READ_ERROR);
        }
    }

    public static void main(String[] args) {
        String publicKeyName="~/rsa/key/id_rsa.pub";
        String privateKeyName="~/rsa/key/id_rsa";
        //生成公钥和私钥
        generateKey(publicKeyName,privateKeyName,"LonelySnow",0);
        //获取公钥和私钥
        PublicKey publicKey = getPublicKey(publicKeyName);
        System.out.println(publicKey);
        PrivateKey privateKey = getPrivateKey(privateKeyName);
        System.out.println(privateKey);

    }
}
