package cn.lonelysnow.common.auth;

import cn.lonelysnow.common.exception.CommonExceptionEnum;
import cn.lonelysnow.common.exception.SnowException;
import cn.lonelysnow.common.utils.json.JacksonUtils;
import io.jsonwebtoken.*;
import io.jsonwebtoken.security.InvalidKeyException;
import io.jsonwebtoken.security.Keys;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.UnsupportedJwtException;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.security.SignatureException;
import java.lang.IllegalArgumentException;

import java.nio.charset.StandardCharsets;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.*;

/**
 * @author LonelySnow
 * @classname JwtUtils
 * @description JWT工具类
 * @date 2022/9/9 15:02
 */
@Slf4j
public class JwtUtils {

    /**
     * 定义jwt默认载荷key
     */
    private static final String JWT_PAYLOAD_USER_KEY = "user";

    /**
     * 定义jwt载荷主体
     */
    private static final String JWT_PAYLOAD_SUBJECT_VALUE = "LonelySnow";

    /**
     * 定义默认有效时间1小时
     */
    private static final Long JWT_DEFAULT_EXPIRATION_TIME = 60 * 60 *1000L;

    /**
     * 定义默认秘钥
     */
    private static final String JWT_DEFAULT_SIGN_KEY = "882d12d5d76d6e0ee3c7d0c076f76ddf9cd04863030c4906d5e2d8711b2fa128";

    /**
     * RSA-私钥生成token(过期时间-分钟)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param privateKey    私钥
     * @param expire        过期时间，分钟
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpireInMinutes(Object userInfo, PrivateKey privateKey, int expire) {
        try {
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(DateTime.now().plusMinutes(expire).toDate())
                    // 指定签名
                    .signWith(privateKey, SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * RSA-私钥生成token(过期时间-秒)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param privateKey    私钥
     * @param expire        过期时间，秒
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpireInSeconds(Object userInfo, PrivateKey privateKey, int expire) {
        try {
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(DateTime.now().plusSeconds(expire).toDate())
                    // 指定签名
                    .signWith(privateKey, SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * RSA-私钥生成token(过期时间-毫秒)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param privateKey    私钥
     * @param expire        过期时间，毫秒
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpire(Object userInfo, PrivateKey privateKey, Long expire) {
        try {
            long nowMillis = System.currentTimeMillis();
            long expMillis = nowMillis + expire;
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(new Date(expMillis))
                    // 指定签名
                    .signWith(privateKey, SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * RSA-私钥生成token(过期时间-毫秒)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param privateKey    私钥
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpire(Object userInfo, PrivateKey privateKey) {
        try {
            long nowMillis = System.currentTimeMillis();
            long expMillis = nowMillis + JWT_DEFAULT_EXPIRATION_TIME;
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(new Date(expMillis))
                    // 指定签名
                    .signWith(privateKey, SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 指定秘钥-生成token(过期时间-分钟)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param keys          加密秘钥(足够长，且复杂)
     * @param expire        过期时间，分钟
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpireInMinutes(Object userInfo, String keys, int expire) {
        try {
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(DateTime.now().plusMinutes(expire).toDate())
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(keys.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (InvalidKeyException e) {
            log.error("Token生成失败，JWT秘钥复杂度未达标：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_SIGN_KEY_ERROR);
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }


    /**
     * 指定秘钥-生成token(过期时间-秒)
     * @author LonelySnow
     * @param userInfo  载荷中的数据
     * @param keys      加密秘钥(足够长，且复杂)
     * @param expire    过期时间，秒
     * @return java.lang.String
     * @date 2022/9/13 11:04
     */
    public static String generateTokenExpireInSeconds(Object userInfo, String keys, int expire) {
        try {
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(DateTime.now().plusSeconds(expire).toDate())
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(keys.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (InvalidKeyException e) {
            log.error("Token生成失败，JWT秘钥复杂度未达标：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_SIGN_KEY_ERROR);
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 指定秘钥-生成token(过期时间-毫秒)
     * @author LonelySnow
     * @param userInfo  载荷中的数据
     * @param keys      加密秘钥(足够长，且复杂)
     * @param expire    过期时间，毫秒
     * @return java.lang.String
     * @date 2022/9/13 11:04
     */
    public static String generateTokenExpire(Object userInfo, String keys, Long expire) {
        try {
            long nowMillis = System.currentTimeMillis();
            long expMillis = nowMillis + expire;
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(new Date(expMillis))
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(keys.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (InvalidKeyException e) {
            log.error("Token生成失败，JWT秘钥复杂度未达标：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_SIGN_KEY_ERROR);
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 指定秘钥-生成token(过期时间-毫秒)
     * @author LonelySnow
     * @param userInfo  载荷中的数据
     * @param keys      加密秘钥(足够长，且复杂)
     * @return java.lang.String
     * @date 2022/9/13 11:04
     */
    public static String generateTokenExpire(Object userInfo, String keys) {
        try {
            long nowMillis = System.currentTimeMillis();
            long expMillis = nowMillis + JWT_DEFAULT_EXPIRATION_TIME;
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(new Date(expMillis))
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(keys.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (InvalidKeyException e) {
            log.error("Token生成失败，JWT秘钥复杂度未达标：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_SIGN_KEY_ERROR);
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 默认秘钥-生成token(过期时间-分钟)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param expire        过期时间，分钟
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpireInMinutes(Object userInfo, int expire) {
        try {
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(DateTime.now().plusMinutes(expire).toDate())
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(JWT_DEFAULT_SIGN_KEY.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 默认秘钥-生成token(过期时间-秒)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param expire        过期时间，秒
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpireInSeconds(Object userInfo, int expire) {
        try {
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(DateTime.now().plusSeconds(expire).toDate())
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(JWT_DEFAULT_SIGN_KEY.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 默认秘钥-生成token(过期时间-毫秒)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @param expire        过期时间，毫秒
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpire(Object userInfo, Long expire) {
        try {
            long nowMillis = System.currentTimeMillis();
            long expMillis = nowMillis + expire;
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(new Date(expMillis))
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(JWT_DEFAULT_SIGN_KEY.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * 默认秘钥-生成token(过期时间-秒)
     * @author LonelySnow
     * @param userInfo      载荷中的数据
     * @return java.lang.String
     * @date 2022/9/13 10:48
     */
    public static String generateTokenExpire(Object userInfo) {
        try {
            long nowMillis = System.currentTimeMillis();
            long expMillis = nowMillis + JWT_DEFAULT_EXPIRATION_TIME;
            return Jwts.builder()
                    // 定义头信息-Map
                    // .setHeader()
                    // 载荷内容
                    .claim(JWT_PAYLOAD_USER_KEY, JacksonUtils.toString(userInfo))
                    // 声明的标识{"jti":""}
                    .setId(createJTI())
                    // 声明主体-String
                    .setSubject(JWT_PAYLOAD_SUBJECT_VALUE)
                    // 创建日期
                    .setIssuedAt(new Date())
                    // 过期时间
                    .setExpiration(new Date(expMillis))
                    // 指定签名
                    .signWith(Keys.hmacShaKeyFor(JWT_DEFAULT_SIGN_KEY.getBytes(StandardCharsets.UTF_8)), SignatureAlgorithm.HS256)
                    // 生成token
                    .compact();
        } catch (Exception e) {
            log.error("Token生成失败，错误原因：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_GENERATE_TOKEN_ERROR);
        }
    }

    /**
     * RSA-校验token是否过期
     * @author LonelySnow
     * @param token     token
     * @param publicKey 公钥
     * @return java.lang.Boolean
     * @date 2022/9/13 11:28
     */
    public static Boolean validateToken(String token, PublicKey publicKey) {
        Date expiredDate = getExpiredDateFromToken(token, publicKey);
        return expiredDate.before(new Date());
    }

    /**
     * 指定秘钥-校验token是否过期
     * @author LonelySnow
     * @param token     token
     * @param keys      秘钥
     * @return java.lang.Boolean
     * @date 2022/9/13 11:28
     */
    public static Boolean validateToken(String token, String keys) {
        Date expiredDate = getExpiredDateFromToken(token, keys);
        return expiredDate.before(new Date());
    }

    /**
     * 默认秘钥-校验token是否过期
     * @author LonelySnow
     * @param token     token
     * @return java.lang.Boolean
     * @date 2022/9/13 11:28
     */
    public static Boolean validateToken(String token) {
        Date expiredDate = getExpiredDateFromToken(token);
        return expiredDate.before(new Date());
    }

    /**
     * RSA-从token中获取过期时间
     * @author LonelySnow
     * @param token         token
     * @param publicKey     公钥
     * @return java.util.Date
     * @date 2022/9/13 11:32
     */
    public static Date getExpiredDateFromToken(String token, PublicKey publicKey) {
        Claims claimsFromToken = getClaimsFromToken(token, publicKey);
        return claimsFromToken.getExpiration();
    }

    /**
     * 指定秘钥-从token中获取过期时间
     * @author LonelySnow
     * @param token         token
     * @param keys          秘钥
     * @return java.util.Date
     * @date 2022/9/13 11:32
     */
    public static Date getExpiredDateFromToken(String token, String keys) {
        Claims claimsFromToken = getClaimsFromToken(token, keys);
        return claimsFromToken.getExpiration();
    }

    /**
     * 默认秘钥-从token中获取过期时间
     * @author LonelySnow
     * @param token         token
     * @return java.util.Date
     * @date 2022/9/13 11:32
     */
    public static Date getExpiredDateFromToken(String token) {
        Claims claimsFromToken = getClaimsFromToken(token);
        return claimsFromToken.getExpiration();
    }

    /**
     * RSA-获取用户信息
     * @author LonelySnow
     * @param token         token
     * @param publicKey     公钥
     * @param userType      信息类(class)
     * @return T
     * @date 2022/9/13 13:16
     */
    public static <T> T getInfoFromToken(String token, PublicKey publicKey, Class<T> userType) {
        Claims body = getClaimsFromToken(token, publicKey);
        return JacksonUtils.toBean(body.get(JWT_PAYLOAD_USER_KEY).toString(), userType);
    }

    /**
     * 指定秘钥-获取用户信息
     * @author LonelySnow
     * @param token         token
     * @param keys          秘钥
     * @param userType      信息类(class)
     * @return T
     * @date 2022/9/13 13:16
     */
    public static <T> T getInfoFromToken(String token, String keys, Class<T> userType) {
        Claims body = getClaimsFromToken(token, keys);
        return JacksonUtils.toBean(body.get(JWT_PAYLOAD_USER_KEY).toString(), userType);
    }

    /**
     * 默认秘钥-获取用户信息
     * @author LonelySnow
     * @param token         token
     * @param userType      信息类(class)
     * @return T
     * @date 2022/9/13 13:16
     */
    public static <T> T getInfoFromToken(String token, Class<T> userType) {
        Claims body = getClaimsFromToken(token);
        return JacksonUtils.toBean(body.get(JWT_PAYLOAD_USER_KEY).toString(), userType);
    }

    /**
     * 创建声明标识符
     * @author LonelySnow
     * @param
     * @return java.lang.String
     * @date 2022/9/13 10:55
     */
    private static String createJTI() {
        return new String(Base64.getEncoder().encode(UUID.randomUUID().toString().getBytes()));
    }

    /**
     * RSA-公钥解析token
     * @author LonelySnow
     * @param token         token
     * @param publicKey     公钥
     * @return io.jsonwebtoken.Jws<io.jsonwebtoken.Claims>
     * @date 2022/9/13 11:23
     */
    private static Jws<Claims> parserToken(String token, PublicKey publicKey) {
        try {
            JwtParser jwtParser = Jwts.parserBuilder()
                    .setSigningKey(publicKey)
                    .build();
            return jwtParser.parseClaimsJws(token);
        } catch (ExpiredJwtException e) {
            log.info("Token已失效：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_EXPIRED);
        } catch (UnsupportedJwtException e) {
            log.info("非本站签发的Token：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_UNSUPPORTED);
        } catch (MalformedJwtException e) {
            log.info("格式错误的Token：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_MALFORMED);
        } catch (SignatureException e) {
            log.info("签名异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_SIGN);
        } catch (IllegalArgumentException e) {
            log.info("非法参数异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_ILLEGAL_ARGUMENT);
        } catch (Exception e) {
            log.info("Token解析失败：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR);
        }
    }

    /**
     * 指定秘钥-解析token
     * @author LonelySnow
     * @param token         token
     * @param keys          加密秘钥(足够长，且复杂)
     * @return io.jsonwebtoken.Jws<io.jsonwebtoken.Claims>
     * @date 2022/9/13 11:23
     */
    private static Jws<Claims> parserToken(String token, String keys) {
        try {
            JwtParser jwtParser = Jwts.parserBuilder()
                    .setSigningKey(Keys.hmacShaKeyFor(keys.getBytes(StandardCharsets.UTF_8)))
                    .build();
            return jwtParser.parseClaimsJws(token);
        } catch (ExpiredJwtException e) {
            log.info("Token已失效：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_EXPIRED);
        } catch (UnsupportedJwtException e) {
            log.info("非本站签发的Token：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_UNSUPPORTED);
        } catch (MalformedJwtException e) {
            log.info("格式错误的Token：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_MALFORMED);
        } catch (SignatureException e) {
            log.info("签名异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_SIGN);
        } catch (IllegalArgumentException e) {
            log.info("非法参数异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_ILLEGAL_ARGUMENT);
        } catch (Exception e) {
            log.info("Token解析失败：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR);
        }
    }

    /**
     * 默认秘钥-解析token
     * @author LonelySnow
     * @param token         token
     * @return io.jsonwebtoken.Jws<io.jsonwebtoken.Claims>
     * @date 2022/9/13 11:23
     */
    private static Jws<Claims> parserToken(String token) {
        try {
            JwtParser jwtParser = Jwts.parserBuilder()
                    .setSigningKey(Keys.hmacShaKeyFor(JWT_DEFAULT_SIGN_KEY.getBytes(StandardCharsets.UTF_8)))
                    .build();
            return jwtParser.parseClaimsJws(token);
        } catch (ExpiredJwtException e) {
            log.info("Token已失效：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_EXPIRED);
        } catch (UnsupportedJwtException e) {
            log.info("非本站签发的Token：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_UNSUPPORTED);
        } catch (MalformedJwtException e) {
            log.info("格式错误的Token：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_MALFORMED);
        } catch (SignatureException e) {
            log.info("签名异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_SIGN);
        } catch (IllegalArgumentException e) {
            log.info("非法参数异常：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR_ILLEGAL_ARGUMENT);
        } catch (Exception e) {
            log.info("Token解析失败：{}", e.getMessage());
            throw new SnowException(CommonExceptionEnum.JWT_PARSER_TOKEN_ERROR);
        }
    }

    /**
     * RSA-从Token中获取载荷
     * @author LonelySnow
     * @param token         token
     * @param publicKey     公钥
     * @return io.jsonwebtoken.Claims
     * @date 2022/9/13 11:38
     */
    private static Claims getClaimsFromToken(String token, PublicKey publicKey) {
        Jws<Claims> claimsJws = parserToken(token, publicKey);
        return claimsJws.getBody();
    }

    /**
     * 指定秘钥-从Token中获取载荷
     * @author LonelySnow
     * @param token     token
     * @param keys      秘钥
     * @return io.jsonwebtoken.Claims
     * @date 2022/9/13 11:38
     */
    private static Claims getClaimsFromToken(String token, String keys) {
        Jws<Claims> claimsJws = parserToken(token, keys);
        return claimsJws.getBody();
    }

    /**
     * 默认秘钥-从Token中获取载荷
     * @author LonelySnow
     * @param token     token
     * @return io.jsonwebtoken.Claims
     * @date 2022/9/13 11:38
     */
    private static Claims getClaimsFromToken(String token) {
        Jws<Claims> claimsJws = parserToken(token);
        return claimsJws.getBody();
    }

    public static void main(String[] args) throws Exception {
        String publicKeyName="D:\\worksoft\\javaee117\\ssh\\id_rsa.pub";
        String privateKeyName="D:\\worksoft\\javaee117\\ssh\\id_rsa";
        PublicKey publicKey = RsaUtils.getPublicKey(publicKeyName);
        PrivateKey privateKey =  RsaUtils.getPrivateKey(privateKeyName);

        Map<String, Object> userInfo = new HashMap<>();
        userInfo.put("id", 1L);
        userInfo.put("username", "LonelySnow");
        userInfo.put("role", "guest");

       /* //生成token
        String token = generateTokenExpireInMinutes(userInfo, privateKey, 3);
        System.out.println(token);*/
        String token="eyJhbGciOiJSUzI1NiJ9.eyJ1c2VyIjoie1wiaWRcIjoxLsdfasasfd5hbWVcIjpcImxleW91XCIsXCJyb2xlXCI6XCJndWVzdFwifSIsImp0aSI6Ik1ESTNNakppWW1NdE16QTFNaTAwTTJJekxXRTFNRFl0Tm1Oa1pqaGlObUkzTVRWbCIsImV4cCI6MTU4MTA2NDM5NX0.h4PZdpjQbXIkbktMJ-SGE77vNFnMcXTMnESL7l6HNGYIvfY29C_k3DmwZSbqRO-cotYEvTgtDMcZur4kW3mVqR-veGWav9lCdT4CWG0Q-0xUPT9ADEVaKfXgzHm4Nrfp9KropIHN_RzDpzlUYFlrhkrn9fpxrTQTy4NN5JdBNLvf4cW10_RHZbng4xOx7CvipTPhAqibQaK4IeqTE5kafXgO4e114VPF4u5Mnpnrav8unYthgsjpPUXb_Hzw8GKAlXI8epfwAUOxsMeyhGfHslnqXTrz8YSOJqyDF4vH_s6cPUuo1B2laMzPP13CqQhx7OY5PpvZZXRi5Vq8dPfpWg";
        //解析token
        Map payload = getInfoFromToken(token, publicKey, Map.class);
        System.out.println(payload);
    }

}
