package cn.lnkdoc.sdk.uia.instance.bccastle.client

import cn.lnkdoc.sdk.uia.common.HttpMethod
import cn.lnkdoc.sdk.uia.common.client.IUiaClient
import cn.lnkdoc.sdk.uia.common.exception.UiaException
import cn.lnkdoc.sdk.uia.common.interceptor.DefaultUiaClientInterceptor
import cn.lnkdoc.sdk.uia.common.request.IUiaRequest
import cn.lnkdoc.sdk.uia.common.response.IUiaResponse
import cn.lnkdoc.sdk.uia.common.response.UiaResponse.Companion.fail
import cn.lnkdoc.sdk.uia.common.response.UiaResponse.Companion.success
import cn.lnkdoc.sdk.uia.common.util.Assert.required
import cn.lnkdoc.sdk.uia.instance.bccastle.property.BccastleProperty
import cn.lnkdoc.sdk.uia.instance.bccastle.util.CheckResponseUtil.check
import com.alibaba.fastjson2.isJSONObject
import io.vavr.Tuple
import io.vavr.Tuple2
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.Request
import okhttp3.RequestBody
import org.slf4j.LoggerFactory

/**
 * @author langkye
 * @since 1.0.0.RELEASE
 */
@Suppress("Unchecked_cast", "unused")
class BccastleUiaClient private constructor() : IUiaClient {
    /**
     * builder
     */
    private val builder: OkHttpClient.Builder = OkHttpClient().newBuilder()

    /**
     * interceptor
     */
    private val interceptor: Interceptor = DefaultUiaClientInterceptor()

    /**
     * property
     */
    private lateinit var property: BccastleProperty

    /**
     * execute
     *
     * @param request request implementation
     * @return response implementation`
     * @throws UiaException UiaException
     */
    @Throws(UiaException::class)
    override fun <RESP> execute(request: IUiaRequest): IUiaResponse<RESP> {
        return try {
            // send request
            val string = sendRequest(request)

            // check success
            check(string, property.isPrintStack())

            // convert 
            val converts = request.getConvert<Any, Any>()
            required(converts, "not found converter for [${request.javaClass.getSimpleName()}]")
            val convert = converts[0]
            val resp = convert.convertResponse<Any, String>(string) as RESP

            // check success
            success(resp)
        } catch (e: Exception) {
            if (property.isPrintStack()) {
                log.error("", e)
            }
            fail(e.message)
        }
    }

    private fun sendRequest(request: IUiaRequest): String {
        // build request url
        val url = request.url(property)
        val logMessage = String.format("[%s][%s]", request.method(), url)
        var success = false
        var string = ""
        // match converter
        val convert = request.getConvert<Any, Any>()[0]

        // build httpRequest
        val body = convert.convertRequest<RequestBody, Tuple2<BccastleProperty?, IUiaRequest>>(
            Tuple.of(
                property,
                request
            )
        )

        // build httpRequest
        val httpRequest: Request = Request.Builder()
            .url(url)
            .method(request.method().method, if (HttpMethod.POST == request.method()) body else null)
            .build()
        
        // call 
        try {
            val client = buildClient(property)
            client.newCall(httpRequest).execute().use { response ->

                // fetch request
                val responseBody = response.body
                required(responseBody, "请求无响应内容：[$url]")
                string = responseBody!!.string()
                success = string.isJSONObject()
                return string
            }
        } catch (e: Exception) {
            throw UiaException(e)
        } finally {
            log.debug("{}[{}][{}]", logMessage, success, string)
        }
    }
    
    private fun buildClient(property: BccastleProperty): OkHttpClient {
        if (property.isUseInterceptor()) {
            builder.addInterceptor(interceptor)
        }
        return builder.build();
    }

    companion object {
        private val log = LoggerFactory.getLogger(BccastleUiaClient::class.java)

        /**
         * instance
         *
         * @param property property
         * @return client
         */
        fun getInstance(property: BccastleProperty): BccastleUiaClient {
            val client = BccastleUiaClient()
            client.property = property
            checkMustRequired(property)
            return client
        }

        private fun checkMustRequired(property: BccastleProperty) {
            required(property, "the uiaConfiguration is required")
            required(property.domain, "the domain configuration is required")
            required(property.clientId, "the clientId configuration is required")
            required(property.clientSecret, "the clientSecret configuration is required")
        }
    }
}
