package cn.lingyangwl.framework.mybatis.mate.interceptor;

import com.alibaba.ttl.TransmittableThreadLocal;
import cn.lingyangwl.framework.data.mate.annotations.FieldBind;
import cn.lingyangwl.framework.data.mate.fieldbind.FieldBindHandler;
import cn.lingyangwl.framework.data.mate.fieldbind.inter.IDataBind;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.mapping.SqlCommandType;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Properties;

/**
 * 拦截StatementHandler类中参数类型为Statement的 prepare 方法
 *
 * @author shenguangyang
 */
@Intercepts({
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}),
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class})
})
public class MybatisFieldBindCustomInterceptor implements Interceptor {
    private static final Logger log = LoggerFactory.getLogger(MybatisFieldBindCustomInterceptor.class);

    private final FieldBindHandler fieldBindHandler;

    public MybatisFieldBindCustomInterceptor(FieldBindHandler fieldBindHandler) {
        this.fieldBindHandler = fieldBindHandler;
    }

    /**
     * 存放线程id, 目的是解决在回调 {@link IDataBind} 接口中的方法时候, 如果方法中调用了数据库查询
     * 字典数据, 不巧的是查询的字典集合中对象字段也有标注 {@link FieldBind} 注解, 这样会造成死循环
     * 从而导致数据库连接数过多
     */
    private static final TransmittableThreadLocal<Long> ttl = new TransmittableThreadLocal<>();

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        Object[] args = invocation.getArgs();
        MappedStatement mappedStatement = (MappedStatement) args[0];
        Object parameter = args[1];
        RowBounds rowBounds = (RowBounds) args[2];
        ResultHandler resultHandler = (ResultHandler) args[3];
        Executor executor = (Executor) invocation.getTarget();
        CacheKey cacheKey;
        BoundSql boundSql;

        // 由于逻辑关系，只会进入一次
        if (args.length == 4) {
            // 4 个参数时
            boundSql = mappedStatement.getBoundSql(parameter);
            cacheKey = executor.createCacheKey(mappedStatement, parameter, rowBounds, boundSql);
        } else {
            // 6 个参数时
            cacheKey = (CacheKey) args[4];
            boundSql = (BoundSql) args[5];
        }

        // 执行修改后的sql语句
        List<Object> query = executor.query(mappedStatement, parameter, rowBounds, resultHandler, cacheKey, boundSql);
        Long threadId;
        if (((threadId = ttl.get()) != null) && threadId == Thread.currentThread().getId()) {
            return query;
        }
        ttl.set(Thread.currentThread().getId());
        try {
            fieldBindHandler.handleField(query);
        } catch (Exception e) {
            log.error("field bind fail, error: ", e);
        } finally {
            ttl.remove();
        }

        return query;
    }

    @Override
    public Object plugin(Object obj) {
        return Plugin.wrap(obj, this);
    }

    @Override
    public void setProperties(Properties arg0) {
        // doSomething
    }

    private String getOperateType(Invocation invocation) {
        final Object[] args = invocation.getArgs();
        MappedStatement ms = (MappedStatement) args[0];
        SqlCommandType commondType = ms.getSqlCommandType();
        if (commondType.compareTo(SqlCommandType.SELECT) == 0) {
            return "select";
        }
        if (commondType.compareTo(SqlCommandType.INSERT) == 0) {
            return "insert";
        }
        if (commondType.compareTo(SqlCommandType.UPDATE) == 0) {
            return "update";
        }
        if (commondType.compareTo(SqlCommandType.DELETE) == 0) {
            return "delete";
        }
        return null;
    }
}
