package cn.lazytool.json;

import cn.lazytool.core.collection.CollectionTools;
import cn.lazytool.core.util.ArrayTools;
import cn.lazytool.core.util.StringTools;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Set;

/**
 * json 工具类
 * @author fenqing
 * @version 0.0.1
 */
public class JSONTools {


    private JSONTools() {
    }

    /**
     * 转换为JSON字符串
     * @param obj 被转为JSON的对象
     * @return JSON字符串
     */
    public static String toJsonStr(Object obj) {
        if (null == obj) {
            return null;
        }
        if (obj instanceof CharSequence) {
            //如果是字符串，直接返回字符串
            return StringTools.str((CharSequence) obj);
        }
        return toJsonStr(parse(obj));
    }

    public static String toJsonStr(JSON json){
        StringBuilder sb = new StringBuilder();
        if(json instanceof JSONObject){
            JSONObject jsonObject = (JSONObject) json;
            sb.append("{");
            final Set<String> keySet = jsonObject.keySet();
            int len = keySet.size();
            CollectionTools.forEach(keySet, (item, index) -> {
                String key = StringTools.format("\"{}\"", item);
                final Object o = jsonObject.get(item);
                sb.append(key).append(":");
                if(isJsonBaseType(o)){
                    sb.append(baseTypeToJsonStr(o));
                } else {
                    sb.append(toJsonStr(o));
                }
                if(index < len - 1){
                    sb.append(',');
                }
            });
            sb.append("}");
        } else if (json instanceof JSONArray) {
            JSONArray jsonArray = (JSONArray) json;
            sb.append("[");
            int len = jsonArray.size();
            CollectionTools.forEach(jsonArray, (item, index) -> {
                if(isJsonBaseType(item)){
                    sb.append(baseTypeToJsonStr(item));
                } else {
                    sb.append(toJsonStr(item));
                }
                if(index < len - 1){
                    sb.append(',');
                }
            });
            sb.append("]");
        }
        return sb.toString();
    }

    /**
     * 基本类型转json字符串
     * @author fenqing
     * @param obj 对象
     * @return java.lang.String
     */
    private static String baseTypeToJsonStr(Object obj){
        if(isJsonBaseType(obj)){
            if(obj == null){
                return "null";
            } else if(obj instanceof CharSequence) {
                return StringTools.format("\"{}\"", obj.toString());
            } else if(obj instanceof Number){
                if(obj instanceof Double || obj instanceof Float || obj instanceof BigDecimal){
                    return String.valueOf(((Number) obj).doubleValue());
                } else {
                    return String.valueOf(((Number) obj).longValue());
                }
            } else if (obj instanceof Boolean) {
                return obj.toString();
            } else if(obj instanceof Date){
                return String.valueOf(((Date) obj).getTime());
            }
        }
        throw new ClassCastException("对象不是基本类型");
    }

    /**
     * 转换对象为JSON
     * 支持的对象
     * String: 转换为相应的对象
     * Array Collection：转换为JSONArray
     * Bean对象：转为JSONObject
     *
     * @param obj 对象
     * @return JSON
     */
    public static JSON parse(Object obj) {
        //基本类型不可被转换
        if(isJsonBaseType(obj)){
            throw new ClassCastException("此类型不可被转换");
        }
        if (obj instanceof JSON) {
            //如果对象已经是JSON类型，就直接返回
            return (JSON) obj;
        } if(isJsonArray(obj)){
            return parseArray(obj);
        } else {
            return parseObject(obj);
        }
    }

    /**
     * 转换成JSONArray
     * @param obj 对象
     * @return JSONArray
     */
    public static JSONArray parseArray(Object obj){
        return new JSONArray(obj);
    }

    /**
     * 转换成JSONArray
     * @param obj 对象
     * @return JSONArray
     */
    public static JSONObject parseObject(Object obj){
        return new JSONObject(obj);
    }

    /**
     * 判断对象是否 可以转为 json 数组
     * @param obj 对象
     * @author fenqing
     * @return boolean 判断结果
     */
    private static boolean isJsonArray(Object obj){
        return ArrayTools.isArray(obj) || CollectionTools.isIterable(obj);
    }

    /**
     * 是否为 json 基本类型:
     * Number, Boolean, CharSequence, Date, null
     * @author fenqing
     * @param obj 对象
     * @return boolean 结果
     */
    protected static boolean isJsonBaseType(Object obj){
        return obj == null
                || obj instanceof CharSequence
                || obj instanceof Number
                || obj instanceof Date
                || obj instanceof Boolean;
    }

}
