package cn.langpy.kotime.service;

import cn.langpy.kotime.handler.AuthCredential;
import cn.langpy.kotime.model.ExceptionNode;
import cn.langpy.kotime.model.MethodNode;
import cn.langpy.kotime.util.Common;
import cn.langpy.kotime.grpc.api.DataServiceGrpc;
import cn.langpy.kotime.grpc.api.RpcExceptionRelation;
import cn.langpy.kotime.grpc.api.RpcMethodRelation;
import cn.langpy.kotime.grpc.api.RpcParamAnalyse;
import cn.langpy.kotime.util.Context;
import io.grpc.ManagedChannel;
import io.grpc.ManagedChannelBuilder;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.reflect.Parameter;
import java.nio.charset.Charset;
import java.util.Base64;
import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;

@Component
public class DefaultGraphService implements GraphService {
    private static Logger log = Logger.getLogger(DefaultGraphService.class.toString());

    @Value("${ko-time.data-host:localhost}")
    private String host;
    @Value("${ko-time.data-port:9906}")
    private Integer port;

    @Value("${ko-time.data-prefix:}")
    private String dataPrefix;
    @Value("${server.servlet.context-path:}")
    private String serverContext;
    @Value("${server.port:8080}")
    private Integer serverPort;
    @Value("${ko-time.data-token:}")
    private String token;


    private ManagedChannel channel;
    private DataServiceGrpc.DataServiceFutureStub futureStub;

    private AuthCredential authCredential;

    public DefaultGraphService() {
    }

    public synchronized boolean connectServer() {
        authCredential = new AuthCredential(createClientName(), token.trim());
        try {
            channel = ManagedChannelBuilder
                    .forAddress(host, port)
                    .usePlaintext()
                    .build();
            log.info("kotime=>KoTime Server was connected. host=" + host + " port=" + port);
            futureStub = DataServiceGrpc.newFutureStub(channel);
        } catch (Exception e) {
            e.printStackTrace();
            log.severe("No server found.");
            return false;
        }
        return true;
    }

    private String getProjectName() {
        if (!StringUtils.hasText(dataPrefix)) {
            if (StringUtils.hasText(serverContext)) {
                dataPrefix = serverContext.substring(1);
            } else {
                log.severe("kotime=>Undefined ko-time.data-prefix,please define it so that KoTime Server can know the datasource!");
            }
        }
        return dataPrefix;
    }

    private String createClientName() {
        return Base64.getEncoder().encodeToString((getProjectName() + "-" + Context.getIp() + "-" + serverPort+"-"+(StringUtils.hasText(serverContext)?serverContext:"/")+"-"+Context.getRunKey()).getBytes(Charset.defaultCharset()));
    }

    public synchronized boolean checkServer() {
        boolean connectState = true;

        if (channel == null || channel.isTerminated()) {
            connectState = connectServer();
        }
        return connectState;
    }

    @Override
    public void addMethodNode(MethodNode methodNode) {
        if (methodNode == null) {
            return;
        }
        boolean b = checkServer();
        if (!b) {
            return;
        }
        methodNode.setDataSource(getProjectName());
        futureStub.withCallCredentials(authCredential).withDeadlineAfter(2, TimeUnit.SECONDS).addMethodNode(methodNode.toRpc());
    }

    @Override
    public void addParamAnalyse(String methodId, Parameter[] names, Object[] values, double v) {
        if (!StringUtils.hasText(methodId)) {
            return;
        }
        boolean b = checkServer();
        if (!b) {
            return;
        }
        String paramsKey = Common.getPramsStr(names, values);
        RpcParamAnalyse paramAnalyse = RpcParamAnalyse.newBuilder()
                .setDataSource(getProjectName())
                .setMethodId(methodId)
                .setParamsKey(paramsKey)
                .setTimeValue(v)
                .build();
        futureStub.withCallCredentials(authCredential).addParamAnalyse(paramAnalyse);
    }

    @Override
    public void addExceptionNode(ExceptionNode exceptionNode) {
        if (exceptionNode == null) {
            return;
        }
        boolean b = checkServer();
        if (!b) {
            return;
        }
        exceptionNode.setDataSource(getProjectName());
        futureStub.withCallCredentials(authCredential).addExceptionNode(exceptionNode.toRpc());
    }

    @Override
    public void addMethodRelation(MethodNode sourceMethodNode, MethodNode targetMethodNode) {
        if (sourceMethodNode == null || targetMethodNode == null) {
            return;
        }
        boolean b = checkServer();
        if (!b) {
            return;
        }
        sourceMethodNode.setDataSource(getProjectName());
        targetMethodNode.setDataSource(getProjectName());
        RpcMethodRelation methodRelation = RpcMethodRelation.newBuilder()
                .setDataSource(getProjectName())
                .setSource(sourceMethodNode.toRpc())
                .setTarget(targetMethodNode.toRpc())
                .build();
        futureStub.withCallCredentials(authCredential).addMethodRelation(methodRelation);
    }

    @Override
    public void addExceptionRelation(MethodNode sourceMethodNode, ExceptionNode exceptionNode) {
        if (sourceMethodNode == null || exceptionNode == null) {
            return;
        }
        boolean b = checkServer();
        if (!b) {
            return;
        }
        sourceMethodNode.setDataSource(getProjectName());
        exceptionNode.setDataSource(getProjectName());
        RpcExceptionRelation exceptionRelation = RpcExceptionRelation.newBuilder()
                .setDataSource(getProjectName())
                .setSource(sourceMethodNode.toRpc())
                .setTarget(exceptionNode.toRpc())
                .build();
        futureStub.withCallCredentials(authCredential).addExceptionRelation(exceptionRelation);
    }
}
