package cn.langpy.disroute.core;

import cn.langpy.disroute.annotation.Route;
import cn.langpy.disroute.config.SpringContext;
import cn.langpy.disroute.exception.InvalidContextException;
import cn.langpy.disroute.exception.InvalidRoutePathException;
import cn.langpy.disroute.exception.NotInitRouteException;
import com.alibaba.fastjson.JSONObject;
import org.reflections.Reflections;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

public class RouteContext {

    static ConcurrentHashMap<String, Object> instanceMap = new ConcurrentHashMap();
    static ConcurrentHashMap<String, Method> methodMap = new ConcurrentHashMap();
    static ConcurrentHashMap<String, String> keyMap = new ConcurrentHashMap();

    public static boolean isInit() {
        return RouteContext.instanceMap.size() > 0;
    }

    public static void init() {
        init("");
    }

    public static void init(String packagePath) {
        if (isInit()) {
            return;
        }
        Reflections reflections = null;
        if (packagePath != null && packagePath.length() > 1) {
            reflections = new Reflections(packagePath);
        } else {
            reflections = new Reflections();
        }
        Set<Class<?>> sevices = reflections.getTypesAnnotatedWith(Route.class);
        for (Class<?> sevice : sevices) {
            Route contextRoute = sevice.getAnnotation(Route.class);

            String context = contextRoute.value();
            Method[] methods = sevice.getDeclaredMethods();
            for (Method method : methods) {
                Route routePath = method.getAnnotation(Route.class);
                String path = routePath.value();
                String[] pathSplit = path.split("/");
                if (pathSplit.length != 3) {
                    throw new InvalidRoutePathException("error @Route,please refer to @Route(\"/key/value\"");
                }

                String key = "";
                key = pathSplit[1];
                try {
                    String instanceKey  = String.format("%s/%s",context,key);
                    if (!RouteContext.instanceMap.containsKey(instanceKey)) {
                        if (isSpring()) {
                            Component component = sevice.getAnnotation(Component.class);
                            if (component==null) {
                                RouteContext.instanceMap.put(instanceKey, sevice.newInstance());
                            }else {
                                RouteContext.instanceMap.put(instanceKey, SpringContext.applicationContext.getBean(sevice));
                            }
                        }else {
                            RouteContext.instanceMap.put(instanceKey, sevice.newInstance());
                        }
                    }
                    RouteContext.methodMap.put(context + path, method);
                    RouteContext.keyMap.put(context, key);
                } catch (InstantiationException e) {
                    e.printStackTrace();
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    public static boolean isSpring() {
        try {
            Thread.currentThread().getContextClassLoader().loadClass("org.springframework.stereotype.Component");
        } catch (ClassNotFoundException e) {
            return false;
        }
        return true;
    }


    public static String getKey(String context) {
        String key = RouteContext.keyMap.get(String.format("/%s", context));
        if (key == null) {
            throw new InvalidContextException();
        }
        return key;
    }

    public static Object getValue(String key, Object param) {
        Object keyValue = null;
        if (param instanceof JSONObject) {
            JSONObject map = (JSONObject) param;
            keyValue = map.get(key);
        } else if (param instanceof Map) {
            Map map = (Map) param;
            keyValue = map.get(key);
        } else {
            Field[] fields = param.getClass().getDeclaredFields();
            Optional<Field> keyFieldOption = Arrays.stream(fields).filter(name -> name.getName().equals(key)).findFirst();
            keyFieldOption.orElseThrow(() -> new InvalidRoutePathException());
            Field keyField = keyFieldOption.get();
            keyField.setAccessible(true);
            try {
                keyValue = keyField.get(param);
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }

        return keyValue;
    }

    public static Method getMethod(String context, String key, Object keyValue) {
        Method method = RouteContext.methodMap.get(String.format("/%s/%s/%s", context, key, keyValue));
        if (method == null) {
            method = RouteContext.methodMap.get(String.format("/%s/%s/*", context, key));
            if (method == null) {
                throw new RuntimeException(String.format("can not find @Route(\"/%s/%s\")", key, keyValue));
            }
        }
        return method;
    }

    public static <T> T getResult(Method method, String context, String key, Object param) {
        T result = null;
        Object resultObject = null;
        try {
            resultObject = method.invoke(RouteContext.instanceMap.get(String.format("/%s/%s", context, key)), param);
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        }
        if (resultObject != null) {
            result = (T) resultObject;
        }
        return result;
    }

    public static <T> T dispatch(String context, Object param) {
        if (!isInit()) {
            throw new NotInitRouteException();
        }
        String key = getKey(context);
        Object keyValue = getValue(key, param);
        Method method = getMethod(context, key, keyValue);
        T result = getResult(method, context, key, param);
        Route routePath = method.getAnnotation(Route.class);
        if (routePath.nextContext().length() > 1) {
            return dispatch(routePath.nextContext(), result);
        }
        return result;
    }
}
