package cn.lalaki.pub.tasks;

import net.lingala.zip4j.ZipFile;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.model.enums.CompressionLevel;

import org.gradle.api.DefaultTask;
import org.gradle.api.tasks.TaskAction;
import org.gradle.api.tasks.bundling.AbstractArchiveTask;
import org.gradle.internal.hash.HashCode;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

/**
 * Task that calculates checksums for every non-{@code .asc} file in an archive.
 * <p>
 * Created on yyyy-MM-dd
 *
 * @author thebugmc, lalaki
 * @see #calculateFor
 * @since ChecksumTask
 */
public class ChecksumTask extends DefaultTask {
    private AbstractArchiveTask archive;

    /**
     * Specifies what file to calculate checksums for.
     */
    public void calculateFor(AbstractArchiveTask archive) throws IllegalStateException {
        if (this.archive != null && this.archive != archive) {
            throw new IllegalStateException("Already calculating checksum for " + this.archive);
        }
        dependsOn(archive);
        this.archive = archive;
    }

    @Override
    public String toString() {
        return "Checksum for " + archive;
    }

    private byte[] hex(byte[] bytes) {
        return HashCode.fromBytes(bytes).toString().getBytes(StandardCharsets.UTF_8);
    }

    /**
     * File in which the checksum output will be present.
     */
    public File outputFile() {
        var archiveFile = archive.getArchiveFile().get().getAsFile();
        var ext = "." + archive.getArchiveExtension().get();
        var name = archiveFile.getName().substring(0, archiveFile.getName().length() - ext.length());
        return new File(archiveFile.getParentFile(), name + "-checksum" + ext);
    }

    private final HashMap<String, String> hashWithExt = new HashMap<>() {{
        put("MD5", ".md5");
        put("SHA-1", ".sha1");
        put("SHA-256", ".sha256");
        put("SHA-512", ".sha512");
        put("", "");
    }};
    private final ZipParameters params = new ZipParameters();

    /**
     * Copy over files and add their checksums to the {@link #outputFile() output file}.
     *
     * @throws IOException If any IO exception occurs.
     */
    @TaskAction
    public void writeChecksums() throws IOException {
        var archiveFile = archive.getArchiveFile().get().getAsFile();
        params.setCompressionLevel(CompressionLevel.ULTRA);
        try (var bundleZip = new ZipFile(outputFile())) {
            try (var zipArchive = new ZipFile(archiveFile)) {
                for (var headers : zipArchive.getFileHeaders()) {
                    if (!headers.isDirectory()) {
                        var fileName = headers.getFileName();
                        try (var fileStream = zipArchive.getInputStream(headers)) {
                            var fileData = fileStream.readAllBytes();
                            hashWithExt.forEach((alg, ext) -> addFileToZip(bundleZip, fileName + ext, alg.isEmpty() ? fileData : getCheckSum(fileData, alg)));
                        }
                    }
                }
            }
        }
    }

    private void addFileToZip(ZipFile zipFile, String name, byte[] data) {
        params.setFileNameInZip(name);
        try (var fileStream = new ByteArrayInputStream(data)) {
            zipFile.addStream(fileStream, params);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private byte[] getCheckSum(byte[] data, String algorithm) {
        try {
            return hex(MessageDigest.getInstance(algorithm).digest(data));
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }
}